package tkchel;

import de.proveo.event.util.efm.groovy.EventParser;
import de.proveo.event.tkchel.model.TKChelEvent;
import de.proveo.event.tkchel.model.CRC8;

import de.proveo.event.util.raw.ByteUtil;
import java.util.*;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 *
 * @author <a href="mailto:jbader@zebra.com">Joachim Bader</a>
 * $Rev: 21304 $
 */
public class TKChelEventParser implements EventParser 
{
	/** logger instance for this class */
	private static final Log log = LogFactory.getLog(TKChelEventParser.class);
	
    /**
     * 
     * @param event the event send form the device
     * @param parsedEventParameters map, should be filled with parsed key value information
     * @param parsedMetaInfo map, should be filled with parsed event meta information
     */
    public void parseEvent(Object event, Map parsedEventParameters, Map metaInfo)
	{
		try
		{
			TKChelEvent tkChelEvent = (TKChelEvent) event;
			
			metaInfo.put("hwsn", Long.toString(event.getSerialNumber()));
			metaInfo.put("hwType", "tkchel");			
			
			final byte[] data = tkChelEvent.getData();
			
			if(data.length==0)
			{
				// it's only an interface ping event
				return;
			}

			final int recordCount = data.length/16;
			for(int index=0;index<recordCount;index++)
			{
				try
				{
					parseRecord(index, data, parsedEventParameters, metaInfo);
				}
				catch(Exception ex)
				{
					log.error("runs for record "+index+" of "+event, ex);
				}
			}
		}
		catch(Exception ex)
		{
			log.error("runs into", ex);
		}
	}
	
	int getRecordStartByteIndex(int recordIndex)
	{
		return recordIndex*16;
	}
	
	void parseRecord(int recordIndex, byte[] data, Map parsedEventParameters, Map metaInfo)
	{
		final long word = ByteUtil.getUnsingedIntegerLSB(data, getRecordStartByteIndex(recordIndex), 4);
		
		// timestamp
		praseTimestamp(recordIndex, word, parsedEventParameters);

		// basic record or addition record
		if(ByteUtil.isBitSet(word, 31))
		{
			parseAdditionalRecord(recordIndex, data, parsedEventParameters, metaInfo);
		}
		else
		{			
			parseBasicRecord(recordIndex, data, parsedEventParameters, metaInfo);
		}		
	}
	
	void praseTimestamp(int recordIndex, Long firstWord, Map parsedEventParameters)
	{
		boolean timestampFaithful = !ByteUtil.isBitSet(firstWord, 30);
		
		if(!timestampFaithful)
		{
			throw new Exception("timestamp not faithful");
		}
		
		parsedEventParameters.put(recordIndex+".timestamp", calculateTimestamp(firstWord & 0x3FFFFFFF));		
		parsedEventParameters.put(recordIndex+".timestamp.faithful", timestampFaithful);		
	}
	
	private static final TS_MIN = 60;
	private static final TS_HOUR = TS_MIN*60;
	private static final TS_DAY = TS_HOUR*24;
	private static final TS_MONTH = TS_DAY*31;
	private static final TS_YEAR = TS_MONTH*12;
	
	/**
	 * Time (30bit) = Seconds + 60 * min + 3600 * hour + 86400 * (day - 1) + 2678400 * (month - 1) + 32140800 * (year - 2009)
	 *
	 * @params maskedTimestamp raw 30bit timestamp from event
	 * @return java milliseconds timestamp
	 */
	long calculateTimestamp(final long maskedTimestamp)
	{
		Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"), Locale.ENGLISH);

		int timestamp = (int) maskedTimestamp;
		
		// Year
		int value = timestamp/TS_YEAR;
		timestamp -= value*TS_YEAR;
		calendar.set(Calendar.YEAR, value+2009);				
		
		// Month
		value = timestamp/TS_MONTH+1;
		timestamp -= (value-1)*TS_MONTH;
		calendar.set(Calendar.MONTH, value-1);		

		// Day
		value = timestamp/TS_DAY+1;
		timestamp -= (value-1)*TS_DAY;
		calendar.set(Calendar.DAY_OF_MONTH, value);		
		
		// Hour
		value = timestamp/TS_HOUR;
		timestamp -= value*TS_HOUR;
		calendar.set(Calendar.HOUR_OF_DAY, value);		

		// Minute
		value = timestamp/TS_MIN;
		timestamp -= value*TS_MIN;
		calendar.set(Calendar.MINUTE, value);		
		
		// Second
		calendar.set(Calendar.SECOND, timestamp);		
		
		// Milisecond
		calendar.set(Calendar.MILLISECOND, 0);		
		
		return calendar.getTimeInMillis();
	}
	
	void parseAdditionalRecord(int recordIndex, byte[] data, Map parsedEventParameters, Map metaInfo)
	{
		final int recordType = ByteUtil.getInteger(data[getRecordStartByteIndex(recordIndex)+4]);

		parsedEventParameters.put(recordIndex+".tkchel.recordType", recordType);
		
		switch(recordType)
		{
			case 1: 
				parseAddRecAnalogData(recordIndex, data, parsedEventParameters);
				break;
				
			case 2: 
				parseAddRecRecordCounters(recordIndex, 1, data, parsedEventParameters);
				break;
				
			case 3: 
				parseAddRecRecordCounters(recordIndex, 3, data, parsedEventParameters);
				break;
				
			case 4:
				parseAddRecMotion(recordIndex, data, parsedEventParameters);
				break;
			
			case 5: 
				parseAddRecRecordCounters(recordIndex, 5, data, parsedEventParameters);
				break;
				
			case 6: 
				parseAddRec1Wire(recordIndex, data, parsedEventParameters);
				break;
			
			case 7: 
				parseAddRecRecordCounters(recordIndex, 7, data, parsedEventParameters);
				break;
				
			case 8: 
				parseAddRecLLS(recordIndex, 1, data, parsedEventParameters);
				break;
				
			case 9: 
				parseAddRecLLS(recordIndex, 5, data, parsedEventParameters);
				break;
			
			case 15: 
				parseAddRecRecordEvents(recordIndex, data, parsedEventParameters);
				break;
			
			default:
				throw new IllegalArgumentException("unkown record type: "+recordType)
			
		}
		
		parseInputs(recordIndex, data, parsedEventParameters);
		parseDeviceStatus(recordIndex, data, parsedEventParameters);
	}
	
	void parseAddRecMotion(int recordIndex, byte[] data, Map parsedEventParameters)
	{
		int raw = ByteUtil.getIntegerLSB(data,getRecordStartByteIndex(recordIndex)+5,3);
		
		// Speed (knots)
		int rawValue = raw & 0x03FFF;
		double value = rawValue * 0.1d;
		parsedEventParameters.put(recordIndex+".speed", value);
		
		// heading (degrees)
		rawValue = raw >> 15;
		parsedEventParameters.put(recordIndex+".heading", rawValue);		
				
		raw = ByteUtil.getIntegerLSB(data,getRecordStartByteIndex(recordIndex)+9,3);
		
		// height
		int heihgtIncreasedAccuracy = ByteUtil.getInteger(data[8]);				
		rawValue = raw & 0x7FFF;
		double height = rawValue+heihgtIncreasedAccuracy*0.04d;
		parsedEventParameters.put(recordIndex+".height", height);

		// satellites
		rawValue = raw & 0x78000;
		rawValue >>= 15;
		parsedEventParameters.put(recordIndex+".satellites", rawValue);
		
		// HDOP
		rawValue = raw & 0xF80000;
		rawValue >>= 19;		
		parsedEventParameters.put(recordIndex+".hdop", rawValue);
	}
	
	void parseAddRecLLS(int recordIndex, final int firstLLSnumber, byte[] data, Map parsedEventParameters)
	{
		int raw = ByteUtil.getIntegerLSB(data,getRecordStartByteIndex(recordIndex)+5,3);
		int sensor = raw & 0x0FFF;
		parsedEventParameters.put(recordIndex+".lls"+firstLLSnumber, sensor);
		
		sensor = raw & 0xFFF000;
		sensor >>= 12;
		parsedEventParameters.put(recordIndex+".lls"+(firstLLSnumber+1), sensor);
		
		raw = ByteUtil.getInteger(data[getRecordStartByteIndex(recordIndex)+8]);
		for(int index=0;index<4;index++)
		{			
			parsedEventParameters.put(recordIndex+".lls"+(firstLLSnumber+index)+".valid", ByteUtil.isBitSet(raw, index));
		}
		
		raw = ByteUtil.getIntegerLSB(data,getRecordStartByteIndex(recordIndex)+9,3);
		sensor = raw & 0x0FFF;
		parsedEventParameters.put(recordIndex+".lls"+(firstLLSnumber+2), sensor);
		
		sensor = raw & 0xFFF000;
		sensor >>= 12;
		parsedEventParameters.put(recordIndex+".lls"+(firstLLSnumber+3), sensor);		
	}
	
	/**
	 * @todo specification missprint? 1-3 bytes, 4-8 bytes
	 * but only 6 bytes colored?
	 * 
	 * An IButton as normaly 8 bytes (1 byte checksum)
	 * 
	 * @todo the implementation is just a guess!!
	 */
	void parseAddRec1Wire(int recordIndex, byte[] data, Map parsedEventParameters)
	{
		String id = "";
		for(int index=getRecordStartByteIndex(recordIndex)+5;index<(getRecordStartByteIndex(recordIndex)+8);index++)
		{
			id += Integer.toString(data[index], 16);
		}
		
		// add checksum
		int calculatedChecksum = CRC8.compute(data,getRecordStartByteIndex(recordIndex)+5,7);
		id += Integer.toString(calculatedChecksum, 16);;
		
		id = id.toUpperCase();
		parsedEventParameters.put(recordIndex+".onewire", id);	
	}
	
	void parseAddRecAnalogData(int recordIndex, byte[] data, Map parsedEventParameters)
	{
		int raw = ByteUtil.getIntegerLSB(data,getRecordStartByteIndex(recordIndex)+5,3);
		
		// analog 1
		int analogRaw = raw & 0x03FF;
		double analog = analogRaw * 0.0098d;
		parsedEventParameters.put(recordIndex+".an1", analog);
		
		// power supply
		analogRaw = raw & 0x0FFC00;
		analogRaw >>= 10;
		analog = analogRaw * 0.017d;
		parsedEventParameters.put(recordIndex+".powersupply.voltage", analog);		
		
		// reserve battery
		analog = ByteUtil.getInteger(data[getRecordStartByteIndex(recordIndex)+8]) * 0.07d;
		parsedEventParameters.put(recordIndex+".powersupply.batteryVoltage", analog);
		
		
		raw = ByteUtil.getIntegerLSB(data,getRecordStartByteIndex(recordIndex)+9,3);
		
		// analog 2
		analogRaw = raw & 0x03FF;
		analog = analogRaw * 0.0235d;
		parsedEventParameters.put(recordIndex+".an2", analog);		
		
		// todo: year??
		analogRaw = raw & 0x01FC00;
		analogRaw >>= 10;
		parsedEventParameters.put(recordIndex+".tkchel.analog.year", analogRaw);
	}
	
	void parseAddRecRecordEvents(int recordIndex, byte[] data, Map parsedEventParameters)
	{
		int raw = ByteUtil.getInteger(data[getRecordStartByteIndex(recordIndex)+5]);
		parsedEventParameters.put(recordIndex+".recordEvent.eventNumber", raw);
	}
	
	void parseAddRecRecordCounters(int recordIndex, final int firstCounterNumber, byte[] data, Map parsedEventParameters)
	{
		int raw = ByteUtil.getIntegerLSB(data,getRecordStartByteIndex(recordIndex)+5,3);
		parsedEventParameters.put(recordIndex+".counter"+firstCounterNumber, raw);		
		
		raw = ByteUtil.getIntegerLSB(data,getRecordStartByteIndex(recordIndex)+9,3);
		parsedEventParameters.put(recordIndex+".counter"+firstCounterNumber+1, raw);		
	}
	
	void parseBasicRecord(int recordIndex, byte[] data, Map parsedEventParameters, Map metaInfo)
	{
		// 28 bit latitude
		long word = ByteUtil.getUnsingedIntegerLSB(data,getRecordStartByteIndex(recordIndex)+4,4);
		word &= 0x0FFFFFFF;
		int latitude = word * 0.00005d / 60d * 60d * 60d * 1000d;
		parsedEventParameters.put(recordIndex+".latitude", latitude);
		
		// bit #32 internal/external
		parsedEventParameters.put(recordIndex+".gps.sourceInternal", ByteUtil.isBitSet(data[getRecordStartByteIndex(recordIndex)+7], 7));
		
		// 29 bit longitutde
		word = ByteUtil.getUnsingedIntegerLSB(data,getRecordStartByteIndex(recordIndex)+8,4);
		word &= 0x1FFFFFFF;
		int longitude = word * 0.00005d / 60d * 60d * 60d * 1000d;
		parsedEventParameters.put(recordIndex+".longitude", longitude);
		 
		// 3 bit HDOP
		parseBasicRecordHDOP(recordIndex, data, parsedEventParameters);
		
		parseInputs(recordIndex, data, parsedEventParameters);
		parseDeviceStatus(recordIndex, data, parsedEventParameters);
	}
	
	/**
	 *Quality coordinates (1 is the best, 7-the worst)
	 * =ln(HDOP * 10) - 1; if 0 - no recieve, 7 - no data about receive
	 * 0 - similar to the flag (V) in the old format
	 *
	 * bits(decimal:0..7) = r = ln(HDOP * 10) - 1
	 * 
	 * HDOP = e^(r+1) / 10
	 */
	void parseBasicRecordHDOP(int recordIndex, byte[] data, Map parsedEventParameters)
	{
		int b =  ByteUtil.getInteger(data[getRecordStartByteIndex(recordIndex)+11]);
		
		// get the highest three bits, b = 0..7
		b >>= 5;
		
		if(b>0)
		{		
			double hdop = Math.exp(b+1)/10.0d;		
			parsedEventParameters.put(recordIndex+".hdop", hdop);
			parsedEventParameters.put(recordIndex+".trusted", Boolean.TRUE);
		}
		else
		{
			parsedEventParameters.put(recordIndex+".trusted", Boolean.FALSE);
		}
	}
	
	void parseInputs(int recordIndex, byte[] data, Map parsedEventParameters)
	{
		final int b = ByteUtil.getInteger(data[getRecordStartByteIndex(recordIndex)+12]);
		
		for(int index=0;index<8;index++)
		{
			parsedEventParameters.put(recordIndex+".in"+index, ByteUtil.isBitSet(b, index));
		}
	}
	
	void parseDeviceStatus(int recordIndex, byte[] data, Map parsedEventParameters)
	{
		int b = ByteUtil.getInteger(data[getRecordStartByteIndex(recordIndex)+13]);
		
		parsedEventParameters.put(recordIndex+".tkchel.server1", ByteUtil.isBitSet(b, 0));
		parsedEventParameters.put(recordIndex+".tkchel.server2", ByteUtil.isBitSet(b, 1));
		
		parsedEventParameters.put(recordIndex+".tkchel.powerSupply", ByteUtil.isBitSet(b, 2));
		parsedEventParameters.put(recordIndex+".tkchel.powerRedundant", ByteUtil.isBitSet(b, 3));
		
		parsedEventParameters.put(recordIndex+".tkchel.antenna1", ByteUtil.isBitSet(b, 4));
		parsedEventParameters.put(recordIndex+".tkchel.antenna2", ByteUtil.isBitSet(b, 5));
				
		b = ByteUtil.getInteger(data[getRecordStartByteIndex(recordIndex)+14]);
		
		parsedEventParameters.put(recordIndex+".tkchel.canRotation", ByteUtil.isBitSet(b, 0));
		parsedEventParameters.put(recordIndex+".gsm.homeNetwork", ByteUtil.isBitSet(b, 1));
		parsedEventParameters.put(recordIndex+".tkchel.canRevolution", b >> 2);
	}
}

