package tkchel;

import de.proveo.wwt.logic.ejb.dataIn.event.mapper.*;

import de.proveo.event.util.efm.EFMEventUtil;
import de.proveo.util.geo.GPSPosition;

import de.proveo.eventbase.EventConstants;

import de.proveo.wwt.logic.ejb.state.CurrentStateFacadeLocal;
import de.proveo.wwt.logic.ejb.state.CurrentStateFacadeStruct;
import de.proveo.wwt.datamodel.geo.GdataCache;

import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Iterator;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.ArrayList;
import java.util.HashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * General Mapper script for geo, keep alive, notification events, ...
 *
 * The script is compatible only with 4.6.1 and above!
 *
 * @author <a href="mailto:jbader@proveo.com">Joachim Bader</a>
 * $Rev: 22427 $
 * 
 * @todo create current_setup events with software version, serailnumber, ...
 */
class TKChelEventMapper implements EventMapper
{
	private static final Log log = LogFactory.getLog(TKChelEventMapper.class);
	
    protected static final MessageFormat twoDigitsFormat = new MessageFormat("{0,number,#0.00}", Locale.US);
	
	private static final boolean IGNORE_MAPPING_ERROR = false;
	
	private static final double DRIVE_MODE_SPEED_THRESHOLD = 2.7d;
	private static final int KEEP_ALIVE_EVENT_INTERVAL = 15*60*1000;
	    
    public List createEFMevents(Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper) throws Exception
    {
		final List efmEvents = new ArrayList();
		final List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");		
		final Map timestampBasedMap = buildTimestampMap(eventParameters);
		eventParameters.put("timestampBasedMap", timestampBasedMap);
		
		try
		{
			if(efmUnitIds==null)
			{
				// todo: seems to buggy!
				long firstTimestamp = findFirstTimestamp(eventParameters);

				Properties zeroconfEvent = new Properties();
				EFMEventUtil.addZeroConfParameters(zeroconfEvent, firstTimestamp, metainfo);
				efmEvents.add(zeroconfEvent);			
			}
			else
			{		
				int recordIndex = 0;
				boolean recordAvailable = true;
				while(recordAvailable)
				{
					recordAvailable = eventParameters.containsKey(recordIndex+".timestamp");
					if(recordAvailable)
					{
						efmEvents.addAll(createEFMeventsFromRecords(recordIndex, eventParameters, metainfo, facadeWrapper));

						Properties keepAliveEvent = createKeepAliveEvent((Long) eventParameters.get(recordIndex+".timestamp"), metainfo, facadeWrapper)
						if(keepAliveEvent!=null)
						{
							efmEvents.add(keepAliveEvent);
						}

						recordIndex++;
					}					
				}
			}
		}
		catch(Exception ex)
        {
            log.error("createEFMevents()", ex);
            throw ex;
        }				
		
		return efmEvents;
    }
	
	Properties createKeepAliveEvent(long firstTimestamp, Map metainfo, FacadeWrapper facadeWrapper)
	{
		List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");
		
		Long lastKeepAlive = facadeWrapper.getKeepAliveOutFacade().getLastKeepAliveTimestamp(efmUnitIds.get(0));
		if(isNewKeepAliveNecessary(lastKeepAlive, firstTimestamp))
		{				
			Properties event = new Properties();
			EFMEventUtil.addGeneralParameters(event, efmUnitIds, firstTimestamp);

			return event;
		}
		else
		{
			return null;
		}
	}
	
	boolean isNewKeepAliveNecessary(Long lastKeepAlive, long eventTimestamp)
	{
		if(lastKeepAlive==null)
		{
			return true;
		}
		else
		{
			long diff = eventTimestamp-lastKeepAlive.longValue();
			return (diff>KEEP_ALIVE_EVENT_INTERVAL);
		}		
	}

	// seems to be buggy!
	long findFirstTimestamp(Map eventParameters)
	{
		List<Long> timestamps = new ArrayList<Long>(eventParameters.keySet());
		Collections.sort(timestamps);	
		return timestamps.get(0);
	}
	
    List createEFMeventsFromRecords(int recordIndex, Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
    {
		try
		{
			List efmEvents = new ArrayList();

			Properties geoEvent = createPositionEvent(recordIndex, eventParameters, metainfo, facadeWrapper);
			if(geoEvent!=null)
			{
				efmEvents.add(geoEvent);
			}

			// status for each digital input
			createStatusEvents(recordIndex, efmEvents, eventParameters, metainfo, facadeWrapper.getCurrentStateFacade());

			// from additional records		
			if(eventParameters.containsKey(recordIndex+".tkchel.recordType"))
			{
				// measurements for Analog data
				createAnalogMeasurments(recordIndex, efmEvents, eventParameters, metainfo);

				// measurements for counter
				//createCounterMeasurements(recordIndex, efmEvents, eventParameters, metainfo);

				// measurement for Speed, direction, height and hdop
				createMotionMeasurements(recordIndex, efmEvents, eventParameters, metainfo);

				// scanman message for one wire
				//createOneWireMessage(recordIndex, efmEvents, eventParameters, metainfo);

				// measurement for LLS sensor
				//createLLSSensorMeasurement(recordIndex, efmEvents, eventParameters, metainfo);
			}

			return efmEvents;
		}
		catch(Exception ex)
		{
			if(IGNORE_MAPPING_ERROR)
			{
				log.error("createEFMeventsFromRecords", ex);
				return Collections.EMPTY_LIST;				
			}
			else
			{
				throw ex;
			}
		}
    }	
	
    void createLLSSensorMeasurement(int recordIndex, List efmEvents, Map eventParameters, Map metainfo)
    {
		int recordType = eventParameters.get(recordIndex+".tkchel.recordType");
		int firstSensorNumber;
		
		switch(recordType)
		{
			case 8:
			firstSensorNumber = 1; break;
			case 9:
			firstSensorNumber = 5; break;
			
			default:
			return;
		}
		
		List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");

		Properties event = new Properties();
		EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get(recordIndex+".timestamp"));
		
		// 1 lls
		int mType = recordType*10;
		Number mValue = eventParameters.get(recordIndex+".lls"+firstSensorNumber);
		Properties mEvent = event.clone();
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(mType))
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValue);			
		efmEvents.add(mEvent);

		// 2 lls
		mType = recordType*10+1;
		mValue = eventParameters.get(recordIndex+".lls"+(firstSensorNumber+1));
		mEvent = event.clone();
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(mType))
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValue);			
		efmEvents.add(mEvent);
		
		// 3 lls
		mType = recordType*10+2;
		mValue = eventParameters.get(recordIndex+".lls"+(firstSensorNumber+2));
		mEvent = event.clone();
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(mType))
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValue);			
		efmEvents.add(mEvent);

		// 4 lls
		mType = recordType*10+3;
		mValue = eventParameters.get(recordIndex+".lls"+(firstSensorNumber+3));
		mEvent = event.clone();
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(mType))
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValue);			
		efmEvents.add(mEvent);
    }
	
    void createCounterMeasurements(int recordIndex, List efmEvents, Map eventParameters, Map metainfo)
    {
		int recordType = eventParameters.get(recordIndex+".tkchel.recordType");
		
		switch(recordType)
		{
			case 2:
			case 3:
			case 5:
			case 7:
			break;
			
			default:
			return;
		}
		
		List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");

		Properties event = new Properties();
		EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get(recordIndex+".timestamp"));
		
		// lower counter
		int mType = recordType*10;
		Number mValue = eventParameters.get(recordIndex+".counter"+recordType);
		Properties mEvent = event.clone();
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(mType))
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValue);			
		efmEvents.add(mEvent);

		// higher counter
		mType = recordType*10+1;
		mValue = eventParameters.get(recordIndex+".counter"+(recordType+1));
		mEvent = event.clone();
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(mType))
		mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValue);			
		efmEvents.add(mEvent);
    }
	
    void createOneWireMessage(int recordIndex, List efmEvents, Map eventParameters, Map metainfo)
    {
		if(eventParameters.get(recordIndex+".tkchel.recordType").equals(6))
		{
			List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");

			Properties event = new Properties();
			EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get(recordIndex+".timestamp"));
			
			event.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_TYPE, 5);
			event.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO, eventParameters.get(recordIndex+".onewire"));
			
			efmEvents.add(event);
		}
    }
	
    void createOverspeedStatusEvent(double speed, List efmEvents, Properties event)
    {
        int status;
        
        if(speed<30d)
        {
            status = 5001;
        }
        else if(speed<40d)
        {
            status = 5010;
        }
        else
        {
            status = 5040;
        }
        
        event.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, "5000");
        event.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(status));
        efmEvents.add(event);
    }
       
    void createMotionMeasurements(int recordIndex, List efmEvents, Map eventParameters, Map metainfo)
    {
		if(eventParameters.get(recordIndex+".tkchel.recordType").equals(4))
		{
			List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");

			Properties event = new Properties();
			EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get(recordIndex+".timestamp"));

            // speed (km/h) Overspeed warning
			Double speed = eventParameters.get(recordIndex+".speed");
            speed *= 1.852d;            
            createOverspeedStatusEvent(speed, efmEvents, event.clone());            									
		}
    }
		
    void createAnalogMeasurments(int recordIndex, List efmEvents, Map eventParameters, Map metainfo)
    {
		if(eventParameters.get(recordIndex+".tkchel.recordType").equals(1))
		{
			List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");

			Properties event = new Properties();
			EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get(recordIndex+".timestamp"));

			// an2
			/*			mType = 11;
			mValue = eventParameters.get(recordIndex+".an2");
			mEvent = event.clone();
			mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(mType));
			mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValue);
			efmEvents.add(mEvent);*/
			
			// supply voltage
			int mType = 12;
			Number mValue = eventParameters.get(recordIndex+".powersupply.voltage");
			Properties mEvent = event.clone();
            Object[] objArr = new Object[1];
            objArr[0] = mValue;
            String mValueStr = twoDigitsFormat.format(objArr);
			mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(mType));
			mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValueStr);
			efmEvents.add(mEvent);
			
			// battery voltage
			/*			mType = 13;
			mValue = eventParameters.get(recordIndex+".powersupply.batteryVoltage");
			mEvent = event.clone();
			mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(mType));
			mEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValue);
			efmEvents.add(mEvent);*/
			
			// todo: year?????			
		}
    }
	
	boolean isStatusUpdateNecessary(CurrentStateFacadeStruct currentState, int newStatus)
	{
		if(currentState==null)
		{
			return true;
		}
		
		if(currentState.getStateId()!=newStatus)
		{
			// runs unnecessarily -> off
			if((currentState.getStateId()==2013) && (newStatus==2011))
			{
				return true;
			}
			// off -> runs necessarily
			else if((currentState.getStateId()==2011) && (newStatus==2012))
			{
				return true;
			}
			// runs necessarily -> off
			else if((currentState.getStateId()==2012) && (newStatus==2011))
			{
				return true;
			}
		}
		
		return false;
	}
	
	boolean isEngineRunning(int recordIndex, Map eventParameters)
	{
		return eventParameters.get(recordIndex+".in0").equals(Boolean.TRUE);
	}
	
	String toStringStateId(CurrentStateFacadeStruct currentStateStruct)
	{
		if(currentStateStruct==null)
		{
			return "null";
		}
		else
		{
			return currentStateStruct.getStateId();
		}
	}
	
    void createStatusEvents(int recordIndex, List efmEvents, Map eventParameters, Map metainfo, CurrentStateFacadeLocal currentStateFacade)
    {
		List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");
		
		Properties event = new Properties();
		EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get(recordIndex+".timestamp"));

		Properties motorStatusEvent;
		
		// in1
		int stateModel = 2010;                                    // runs necessarily : off
		int motorStatus = isEngineRunning(recordIndex, eventParameters)?2012:2011;
		
		CurrentStateFacadeStruct currentMotorState = currentStateFacade.getCurrentState(efmUnitIds.get(0), 2010);
		if(log.isTraceEnabled())
		{
			log.trace("untid: "+efmUnitIds+ " currentMotorState:"+currentMotorState.getStateId()+" newMotorStatus="+motorStatus);
		}
		
		if(isStatusUpdateNecessary(currentMotorState, motorStatus)) 
		{
			motorStatusEvent = event.clone();
			motorStatusEvent.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, Integer.toString(stateModel));
			motorStatusEvent.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(motorStatus));			
		}		
		
		// in2
		stateModel = 2050; // operation status model
		CurrentStateFacadeStruct currentOperationState = currentStateFacade.getCurrentState(efmUnitIds.get(0), stateModel);		
		
		int operationStatus;
		if(eventParameters.get(recordIndex+".in5").equals(Boolean.TRUE))
		{
			operationStatus = 2053; // work mode
		}
		else
		{
			Double speed = eventParameters.get(recordIndex+".speed");
			if((speed!=null) && (speed>DRIVE_MODE_SPEED_THRESHOLD) && (motorStatus!=2011))
			{
				operationStatus = 2052; // drive mode
			}
			else
			{
				operationStatus = 2054; // available mode
			}
		}
		
		log.trace("untid: "+efmUnitIds+ " currentOperationState:"+toStringStateId(currentOperationState)+ " newOperationStatus: "+operationStatus);
		
		// if opration status is work or drive
		if((operationStatus==2053) || (operationStatus==2052))
		{
			
			log.trace("untid: "+efmUnitIds+ " operationStatus: "+operationStatus+ " currentMotorState: "+toStringStateId(currentMotorState));
			
			// and motor is running unnecessarily and is still running
			if((currentMotorState!=null) && (currentMotorState.getStateId()==2013) && (isEngineRunning(recordIndex, eventParameters)))
			{
				log.trace("untid: "+efmUnitIds+ " switch to runs necessarily");
				
				// then switch to runs necessarily
				motorStatusEvent = event.clone();
				motorStatusEvent.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, Integer.toString(2010));
				motorStatusEvent.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(2012));							
			}			
		}
		
		if(motorStatusEvent!=null) 
		{
			efmEvents.add(motorStatusEvent);			
		}		
		
		if((currentOperationState==null) || (currentOperationState.getStateId()!=operationStatus))
		{
			Properties operationStatusEvent = event.clone();
			operationStatusEvent.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, Integer.toString(stateModel));
			operationStatusEvent.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(operationStatus));
			efmEvents.add(operationStatusEvent);		
		}
		
		// in3
		/*stateModel = 2020;
		status = eventParameters.get(recordIndex+".in3").equals(Boolean.TRUE)?2022:2021;
		statusEvent = event.clone();
		statusEvent.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, Integer.toString(stateModel));
		statusEvent.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(status));
		efmEvents.add(statusEvent);*/				
    }
	
	
    Properties createPositionEvent(int recordIndex, Map<String, Object> eventParameters, Map<String, Object> metainfo, FacadeWrapper facadeWrapper)
    {
		if(!eventParameters.containsKey(recordIndex+".latitude"))
		{
			return null;
		}
		
		List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");
		
		
		Long timestamp = (Long) eventParameters.get(recordIndex+".timestamp");
		Map<String, Map<String, Object>> timestampBasedMap = eventParameters.get("timestampBasedMap");		
		Map<String, Object> params = timestampBasedMap.get(timestamp);
		
		Properties geoEvent = new Properties();
		EFMEventUtil.addGeneralParameters(geoEvent, efmUnitIds, timestamp);		
		
		boolean trusted = (Boolean) params.get("trusted");
		if(trusted)
		{
			geoEvent.setProperty(EventConstants.ATTRIBUTE_GEO_TRUSTED, "1");
			
			generatePositionRecords(recordIndex, params, eventParameters);

			geoEvent.setProperty(EventConstants.ATTRIBUTE_GEO_RMC, (String) eventParameters.get(recordIndex+".rmc"));		
			geoEvent.setProperty(EventConstants.ATTRIBUTE_GEO_GGA, (String) eventParameters.get(recordIndex+".gga"));			
			
			
			GPSPosition newPosition = new GPSPosition((String) eventParameters.get(recordIndex+".rmc"), (String) eventParameters.get(recordIndex+".gga"));
			
			// Is position update necessary?
			if(!isPositionUpdateNecessary(recordIndex, eventParameters, efmUnitIds, newPosition, facadeWrapper))
			{
				geoEvent = null;
			}
		}
		else
		{
			geoEvent.setProperty(EventConstants.ATTRIBUTE_GEO_TRUSTED, "0");
		}
			
		return geoEvent;
    }
	
	/**
	 * The minimum distance in meters if engine is on to fire a geo event.
	 */
	double getMinimumDistanceDuringEngineRunning()
	{
		return 10;
	}

	/**
	 * The minimum distance in meters if engine is off to fire a geo event.
	 */
	double getMinimumDistanceDuringEngineOff()	
	{
		return 20;
	}
	
	/**
	 *
	 *
	 */
	boolean isPositionUpdateNecessary(int recordIndex, Map<String, Object> eventParameters, List<Long> efmUnitIds, GPSPosition newPosition, FacadeWrapper facadeWrapper)
	{	
		GdataCache currentGdataCache = facadeWrapper.getGdataFacade().findCacheByUnitId(efmUnitIds.get(0));		
		if((currentGdataCache==null) || (!currentGdataCache.getTrusted()))
		{
			return true;
		}
		
		GPSPosition currentPosition = new GPSPosition(currentGdataCache.getRmcRec(), currentGdataCache.getGgaRec());
		double distance = GPSPosition.getDistance(newPosition, currentPosition);
		
		if(distance < 0)
		{
			// Couldn't calculate a distance! 
			return true;
		}
		
		double minDistance;		
		if(isEngineRunning(recordIndex, eventParameters))
		{
			minDistance = getMinimumDistanceDuringEngineRunning();
		}
		else
		{
			minDistance = getMinimumDistanceDuringEngineOff();
		}

		// fire only if distance is more then threshold
		return (distance>minDistance);
	}
	
	void dumpObject(String desc, Object obj)
	{
		System.err.println(desc+":  Object: "+obj+" Class: "+obj.getClass()+" toString: "+obj.toString());
	}
	
    protected void generatePositionRecords(int recordIndex, Map<String, Object> params, Map<String, Object> eventParameters)
    {		
		Map<String, String> parameters = new HashMap<String, String>();

		parameters.put("time", params.get("timestamp").toString());
		parameters.put("longitude", params.get("longitude").toString());
		parameters.put("latitude", params.get("latitude").toString());
		
		if(params.get("hdop")!=null)
		{
			parameters.put("hdop", params.get("hdop").toString());
		}

		if(params.get("speed")!=null)
		{
            Object[] objArr = new Object[1];
            objArr[0] = params.get("speed");
            String mValueStr = twoDigitsFormat.format(objArr);
			
			parameters.put("speed", mValueStr);
		}
				
		if(params.get("heading")!=null)
		{
			parameters.put("trackMadeGood", params.get("heading").toString());
		}
		
		if(params.get("height")!=null)
		{
            Object[] objArr = new Object[1];
            objArr[0] = params.get("height");
            String mValueStr = twoDigitsFormat.format(objArr);			
			
			parameters.put("altitude", mValueStr);
		}				
		
		if(params.get("satellites")!=null)
		{
			parameters.put("sats", params.get("satellites").toString());
		}
		
		try
		{
	 		String rmc = GPSPosition.createNMEArecord("RMC", parameters);
			eventParameters.put(recordIndex+".rmc", rmc);
		}
		catch(Exception ex)
		{
			log.error("generatePositionRecords() rmc", ex);			
		}
		
		try
		{
			String gga = GPSPosition.createNMEArecord("GGA", parameters);
			eventParameters.put(recordIndex+".gga", gga);
		}
		catch(Exception ex)
		{
			log.error("generatePositionRecords() gga", ex);			
		}
    }
	
    
     /**
	  * Creating two Maps out of the incomming eventParameters 
	  * 1. Map: keys=timestamps
	  *		  values=eventcounters
	  * 2. Map: keys=eventcounters
	  *		  values=events
	  * After that runs buildfinishedMap for getting the reqired Map
	  * 
	  * @params eventParameters the parsed event parameters
	  * <table>
	  *   <tr><th>key</th><th>value</th></tr>
	  *   <tr><td>0.abc</td><td>www</th></tr>
	  *   <tr><td>0.timestamp</td><td>123</th></tr>
	  *   <tr><td>2.xzy</td><td>aaa</th></tr>
	  *   <tr><td>2.timestamp</td><td>234</th></tr>
	  *   <tr><td>[record number].[record param name]</td><td>[value]</th></tr>
	  * </table>
	  * 
	  * @return a timestamp based map
	  * <table>
	  *   <tr><th>timestamp</th><th>map</th></tr>
	  *   <tr><td>123</td><td>timestamp=123; abc=www</th></tr>
	  *   <tr><td>234</td><td>timestamp=234; xzy=aaa</th></tr>
	  *   <tr><td>[timestamp]</td><td>[map of key value pairs]</th></tr>
	  * </table>
	  */
    Map buildTimestampMap(Map<String, Object> eventParameters)
	{
		/* build record map
		 * [record number]= Map
		 *                  [record param name]=[record param value]
		 *                  [record param name]=[record param value]
		 */ 
		Map<Long, Map<String, Object>> recordMaps = new HashMap<Long, Map<String, Object>>();
		
		Iterator<String> reocrdParamIterator = eventParameters.keySet().iterator();
		while(reocrdParamIterator.hasNext())
		{
			String recordNumber_paramName = (String) reocrdParamIterator.next();
			
			if(!"timestampBasedMap".equals(recordNumber_paramName))
			{			
				int index = recordNumber_paramName.indexOf(".");

				int recordNumber = Integer.parseInt(recordNumber_paramName.substring(0, index));
				String recordParamName = recordNumber_paramName.substring(index+1);			

				Map<String, Object> recordMap = recordMaps.get(recordNumber);
				if(recordMap==null)
				{
					recordMap = new HashMap<String, Object>();
					recordMaps.put(recordNumber, recordMap);
				}

				recordMap.put(recordParamName, eventParameters.get(recordNumber_paramName));
			}
		}
		
		/* convert to a timestamp based map
		 * [teimstamp]= Map
		 *                  [record param name]=[record param value]
		 *                  [record param name]=[record param value]
		 */		
		Map<Long, Map<String, Object>> timestampMap = new HashMap<Long, Map<String, Object>>();
		Iterator<Map<String, Object>> recordMapIterator = recordMaps.values().iterator();
		while(recordMapIterator.hasNext())
		{
			Map<String, Object> recordParams = recordMapIterator.next();
			long timestamp = recordParams.get("timestamp");
			
			Map<String, Object> existingRecordParams = timestampMap.get(timestamp);
			if(existingRecordParams!=null)
			{
				recordParams.putAll(existingRecordParams);
			}
			
			timestampMap.put(timestamp, recordParams);			
		}
		
		return timestampMap;
    }
}

