package tkchel;

import de.proveo.wwt.logic.ejb.dataIn.event.mapper.*;

import de.proveo.event.util.efm.EFMEventUtil;
import de.proveo.util.geo.GPSPosition;

import de.proveo.eventbase.EventConstants;

import de.proveo.wwt.logic.ejb.state.CurrentStateFacadeLocal;
import de.proveo.wwt.logic.ejb.state.CurrentStateFacadeStruct;

import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Iterator;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.ArrayList;
import java.util.HashMap;

/**
 * Create Fuel Measurement by Analog input.
 * Use an1 and an2.
 *
 * @author <a href="mailto:jbader@proveo.com">Joachim Bader</a>
 * $Rev: 21140 $
 */
class TKChelAnalogFuelLevelMapping implements EventMapper
{
	/**
	 * Set to appropriate measurement type
	 */
	private static final int MEASUREMENT_TYPE = 310;
	
	/**
	 * Maximum tank capacity in liters
	 */
	private static final float MAX_LITERS = 460.0f;
	
	/**
	 * Fuel level calculation 
	 * @return return the fuel level in liters; null if you don't want any event
	 */
	Float calculateLiquidLevelLiterAnalog(Integer value1, Integer value2)
	{
		//The tank is empty: 46 * 0.0098 = 0, 4508V
		//Tank full: 410 * 0.0098 = 4, 018V            
		//(The DATABASE stores information about a large number of our block: 0.0098V for a bit.)			
		final double minimum = 0.4508d;
		final double maximum = 3.895d;			
		
		Float measurement = value1;
		measurement -= minimum;

		// Tank fuel is 460 liters plus 10% expansion (when is hot).
		Float measurementLiters = MAX_LITERS/(maximum-minimum)*measurement;
		
		return measurementLiters;
	}	
	
	// do not modify anything below this line!!!
	// --------------------------------------------------------------------------
	
	private static final MessageFormat twoDigitsFormat = new MessageFormat("{0,number,#0.00}", Locale.US);
	
	/**
	 * @return the fuel level in percentage (0.0-1.0); return null if you don't want a fuel level status event
	 */
	Float calculateFuelLevelPercentage(float fuelLiters)
	{
		return fuelLiters/MAX_LITERS;
	}	
	
    public List createEFMevents(Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{		
		final List efmEvents = new ArrayList();
		final List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");		
		
		if(efmUnitIds==null)
		{
			return null;
		}
		
		int recordIndex = 0;
		boolean recordAvailable = true;
		while(recordAvailable)
		{
			recordAvailable = eventParameters.containsKey(recordIndex+".timestamp");
			if(recordAvailable)
			{
				Integer value1 = getAN(eventParameters, recordIndex, 1);
				Integer value2 = getAN(eventParameters, recordIndex, 2);
				
				Float liquidLevelLiters = calculateLiquidLevelLiterAnalog(value1, value2);
				
				Properties event = createEvent(efmUnitIds, eventParameters, recordIndex, MEASUREMENT_TYPE, liquidLevelLiters);				
				if(event!=null)
				{
					efmEvents.add(event);
				}
				
				if(liquidLevelLiters!=null)
				{
					Float fuelLevelPercentage = calculateFuelLevelPercentage(liquidLevelLiters);
					Properties eventFuelLevelStatus = createEventFuelLevelStatus(efmUnitIds, eventParameters, recordIndex, fuelLevelPercentage);				
					if(eventFuelLevelStatus!=null)
					{
						efmEvents.add(eventFuelLevelStatus);
					}
				}
				
				recordIndex++;
			}					
		}
		
		return efmEvents;
	}
	
	Properties createEventFuelLevelStatus(List<Long> efmUnitIds, Map eventParameters, int recordIndex, Float liquidLevelPercentage)
	{
		if(liquidLevelPercentage==null)
		{
			return null;
		}
		
		Properties event = new Properties();
		EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get(recordIndex+".timestamp"));
		event.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, "2030");
		event.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(getFuelStatus(liquidLevelPercentage)));

		return event;		
	}
	
	int getFuelStatus(Float percentage)
	{
		int fuelStatus;
		if(percentage<0.10)
		{
			fuelStatus = 2030;
		}
		else if(percentage<0.20)
		{
			fuelStatus = 2031;
		}
		else if(percentage<0.30)
		{
			fuelStatus = 2032;
		}
		else if(percentage<0.40)
		{
			fuelStatus = 2033;
		}
		else if(percentage<0.50)
		{
			fuelStatus = 2034;
		}
		else if(percentage<0.60)
		{
			fuelStatus = 2035;
		}
		else if(percentage<0.70)
		{
			fuelStatus = 2036;
		}
		else if(percentage<0.80)
		{
			fuelStatus = 2037;
		}
		else if(percentage<0.90)
		{
			fuelStatus = 2038;
		}
		else // 90-100%
		{
			fuelStatus = 2039;
		}				

		return fuelStatus;
	}
	
	Integer getAN(Map eventParameters, int recordIndex, int number)
	{
		return eventParameters.get(recordIndex+".an"+number);
	}
	
	Properties createEvent(List<Long> efmUnitIds, Map eventParameters, int recordIndex, int measurementType, Float liquidLevelLiters)
	{
		if(liquidLevelLiters==null)
		{
			return null;
		}
		
		Properties event = new Properties();
		EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get(recordIndex+".timestamp"));
		event.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(measurementType));
		Object[] objArr = new Object[1];
		objArr[0] = liquidLevelLiters;
		String mValueStr = twoDigitsFormat.format(objArr);
		event.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValueStr);

		return event;
	}
}