package infomanLite;

import de.proveo.wwt.logic.ejb.dataIn.event.mapper.*;

import de.proveo.event.util.efm.EFMEventUtil;
import de.proveo.util.geo.GPSPosition;

import de.proveo.eventbase.EventConstants;

import java.util.*;

import java.text.*;

import de.proveo.event.util.raw.ByteUtil;

/**
 * Infoman Message Event Mapper.
 * Controls all Infoman Lite debug messages.
 * 
 * Should be used for entry point <code>infoman.event.3rdparty.eventMapper</code>
 * 
 * Events:
 * <ul>
 *	<li>Debug messages -> Infoman Messages</li>
 *	<li>device wakeup/info -> infoman setup</li>
 *	<li>sleep event -> infoman online status</li>
 *	<li>update confirmations -> datasync</li>
 * </ul>
 *
 * @author <a href="mailto:jbader@proveo.com">Joachim Bader</a>
 * $Rev: 22277 $
 */
class InfomanLiteMessageEventMapper implements EventMapper {

	private static final int FT_DEBUG_MSG_FALL_ASLEEP = 56;
	private static final int FT_DEBUG_MSG_CANCEL_SLEEP = 55;	
	
	private static final int FT_DEBUG_MSG_DOTA_ACK = 50;
	private static final int FT_DEBUG_MSG_DOTA_ERROR = 51;
	private static final int FT_DEBUG_MSG_DOTA_DOWNLOAD_ERROR = 52;

	private static final int FT_DEBUG_MSG_ACL_ACK = 60;
	private static final int FT_DEBUG_MSG_ACL_ERROR = 61;
	private static final int FT_DEBUG_MSG_ACL_DOWNLOAD_ERROR = 62;
	
	private static final int FT_DEBUG_MSG = 5;
	private static final int FT_DEBUG_MSG_ACL_ACCESS_DENIED = 1;
		
	private static final int SLEEP_STATE_MODEL = 2110;
	private static final int SLEEP_STATE_ID = 2111;
	
	private static final int SCANMAN_MSG_TYPE_ACL_UDPATE = 32;
	
	public List createEFMevents(Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{
		List<Properties> efmEvents = new ArrayList<Properties>();
		
		if(!eventParameters.isEmpty())
		{
			// device events (wakupe, update, ...)
			createDeviceWakeupEvent(efmEvents, eventParameters, metainfo, facadeWrapper);						
			
			createDeviceSleepEvent(efmEvents, eventParameters, metainfo, facadeWrapper);						
			
			// DataSync Job responses
			handleDataSyncResponseMessage(efmEvents, eventParameters, metainfo, facadeWrapper);
			
			// debug messages
			createDebugMessage(efmEvents, eventParameters, metainfo, facadeWrapper);			

		}
		
		return efmEvents;
	}
	
	void createDeviceSleepEvent(List<Properties> efmEvents, Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{
		if(eventParameters.containsKey("debug.msgType") && ( ((Integer) eventParameters.get("debug.msgType")).equals(FT_DEBUG_MSG_FALL_ASLEEP)   ))
		{
			List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");
			
			// create sleep mode event
			Properties msg = new Properties();
			EFMEventUtil.addGeneralParameters(msg, efmUnitIds, (Long) eventParameters.get("timestamp"));
			
			msg.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, Integer.toString(SLEEP_STATE_MODEL));
			msg.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(SLEEP_STATE_ID));
			
			efmEvents.add(msg);
		}
	}
	
	long getTaskIdFromDebugMessage(Map eventParameters)
	{
		return Long.parseLong(eventParameters.get("debug.msg"));
	}
	
	void handleDataSyncResponseMessage(List<Properties> efmEvents, Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{ 
		if(eventParameters.containsKey("debug.msgType"))
		{
			int msgType = (Integer) eventParameters.get("debug.msgType");
			long taskId;
			switch(msgType)
			{
				// DOTA->DataSync
				case FT_DEBUG_MSG_DOTA_ERROR:	
				case FT_DEBUG_MSG_DOTA_DOWNLOAD_ERROR:
					taskId = getTaskIdFromDebugMessage(eventParameters);
					facadeWrapper.getTaskManagementFacade().cancelTask(taskId, null);
					break;
					
				case FT_DEBUG_MSG_DOTA_ACK:									
					taskId = getTaskIdFromDebugMessage(eventParameters);
					facadeWrapper.getTaskExecutionFacade().finishTask(taskId, null);
					break;
					
				// ACL->OldStyle
				case FT_DEBUG_MSG_ACL_ERROR:
				case FT_DEBUG_MSG_ACL_DOWNLOAD_ERROR:
				case FT_DEBUG_MSG_ACL_ACK:
					handleACLResponseMessage(efmEvents, eventParameters, metainfo, facadeWrapper);
					break;
			}			
		}
	}
	
	void handleACLResponseMessage(List<Properties> efmEvents, Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{
		Properties msg = new Properties();
		List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");
		EFMEventUtil.addGeneralParameters(msg, efmUnitIds, (Long) eventParameters.get("timestamp"));
		msg.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_TYPE, Integer.toString(SCANMAN_MSG_TYPE_ACL_UDPATE));
				
		String token = (String) eventParameters.get("debug.msg");
		int msgType = (Integer) eventParameters.get("debug.msgType");
		if(msgType==FT_DEBUG_MSG_ACL_ACK)
		{
			msg.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO, token);
		}
		else
		{
			int index = 0;
			msg.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+(++index), "error");
			msg.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+(++index), token);
		}
		
		efmEvents.add(msg);
	}
	
	void createDebugMessage(List<Properties> efmEvents, Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{
		if(eventParameters.containsKey("debug.msgType"))
		{
			String msgPrefix = null;
			
			switch(eventParameters.get("debug.msgType"))
			{
				case FT_DEBUG_MSG:
					msgPrefix = "Debug";
					break;
									
				case FT_DEBUG_MSG_ACL_ACCESS_DENIED:
					msgPrefix = "Access Denied";
					break;

				case FT_DEBUG_MSG_DOTA_ACK:
					msgPrefix = "Dota ACK";
					break;

				case FT_DEBUG_MSG_DOTA_ERROR:
					msgPrefix = "Dota Error";
					break;
					
				case FT_DEBUG_MSG_ACL_ACK:
					msgPrefix = "ACL ACK";
					break;

				case FT_DEBUG_MSG_ACL_ERROR:
					msgPrefix = "ACL Error";
					break;
					
				case FT_DEBUG_MSG_CANCEL_SLEEP:
					msgPrefix = "Sleep Cancel";
					break;
				
				case FT_DEBUG_MSG_FALL_ASLEEP:
					msgPrefix = "Sleep";
					break;
			}
			
			List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");
			
			Properties msg = new Properties();
			EFMEventUtil.addGeneralParameters(msg, efmUnitIds, (Long) eventParameters.get("timestamp"));

			msg.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_TYPE, Integer.toString(eventParameters.get("debug.msgType")));
			
			String msgText = msgPrefix;
			if(msgText==null)
			{
				msgText = eventParameters.get("debug.msg");
			}
			else
			{
				msgText += " "+eventParameters.get("debug.msg");
			}
			
			msg.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO, msgText);		
			
			efmEvents.add(msg);
		}
	}
	
	void createDeviceWakeupEvent(List<Properties> efmEvents, Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{
		// Device Start Event
		if(eventParameters.containsKey("device.startStatus"))
		{
			List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");
			
			Properties boot = new Properties();
			EFMEventUtil.addGeneralParameters(boot, efmUnitIds, (Long) eventParameters.get("timestamp"));
			boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_TYPE, "25");

			String msg;
			int startStatus = (Integer) eventParameters.get("device.startStatus");			
			switch(startStatus & 0x07)
			{
				case 0:
					msg = "POWER_ON"; // Normal power on
					break;
				case 1:
					msg = "RESERVED_1";  		    // Reserved
					break;
				case 2:
					msg = "WATCHDOG_RESET";     	// CPU internal CPU watchdog timed out
					break;
				case 3:
					msg = "SOFTWARE_RESET";   	// CPU software reset occurred
					break;
				case 4:
					msg = "HARDWARE_RESET";   	// CPU hardware reset occurred
					break;
				case 5:
					msg = "BROWNOUT_RESET";       // CPU brownout reset occurred
					break;
				case 6:
					msg = "DOWNLOAD_SUCCESS"; 	// Reboot after a successful download process
					break;
				case 7:
					msg = "DOWNLOAD_ERROR";    	// Reboot after an error in download process
					break;					
				default:
					msg = "[unkown]";
			}			
			
			int wakeup = (startStatus >> 8) & 0xFF;
			msg += " - ";			
			switch(wakeup)
			{
				case 0x00:
					msg += "Low-power mode was not activated";
					break;
				case 0x40:	
					msg += "Timeout";
					break;
				case 0x41:	
					msg += "External power supply is present";
					break;
				default:
					if((wakeup >= 0x10) && (wakeup <= 0x14))
					{
						msg += "Condition number: "+(wakeup-0x10);
					}
					else
					{
						msg += "Reason unknown";
					}
			}			
			
			int index = 1;
			boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index,"msg="+msg);
			index++;
			
			if(eventParameters.containsKey("device.productName"))
			{
				boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "arch="+eventParameters.get("device.productName"));
				index++;
				boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "imei="+metainfo.get("hwsn"));
				index++;
				boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "hwSN="+metainfo.get("hwsn"));
                                index++;
                                boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "arch="+eventParameters.get("device.productName")+" "+eventParameters.get("device.hwVersion"));
                                index++;
                                String software = eventParameters.get("device.swVersion");
                                String[] split = software.split(" ");
                                String version;
                                String revision;
                                if(split.length==2){
                                    version = split[0];
                                    revision = split[1];
                                    boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "sw="+version);
                                    index++;
                                    boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "svn="+revision);
                                    index++;                              
                                }
                                else{
                                    boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "sw="+software);
                                }

				boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "modemT="+eventParameters.get("device.modemManufacturer")+" "+eventParameters.get("device.modemType"));
				index++;
				
				boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "modemFW="+eventParameters.get("device.modemSwVersion"));
				index++;
				
				boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "simcNo="+eventParameters.get("device.iccid"));
				index++;
				
				boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "imsi="+eventParameters.get("device.imsi"));
				index++;
				
				boot.setProperty(EventConstants.ATTRIBUTE_SCANMANMESSAGE_INFO+index, "vehicleId="+eventParameters.get("device.vehicleID"));
			}
			else
			{
					boot.setProperty(EventConstants.NO_DEVICE_INFORMATION, "25");
			}
			
			efmEvents.add(boot);
		}		
	}
}