package infomanLite;

import de.proveo.event.util.efm.groovy.EventParser;
import de.proveo.event.infomanLite.model.InfomanLiteEvent;
import de.proveo.event.infomanLite.model.CRC8;
import de.proveo.event.util.raw.ByteArrayLittleEndianParser;
import de.proveo.event.util.raw.ByteArrayBigEndianParser;
import de.proveo.event.util.raw.ByteUtil;

import java.text.NumberFormat;
import java.util.Locale;
import de.proveo.util.geo.GPSPosition;

import java.util.*;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Parser Script for InfomanLite Binary Protocol. See Binary Protocol sepcification
 * for available parameters.
 * 
 * Should be used for entry point <code>infoman.event.infomanLite.parseEvent</code>
 * 
 * @author <a href="mailto:funfried@proveo.com">Fabian Unfried</a>
 * $Rev: 22168 $
 */
public class InfomanLiteEventParser implements EventParser 
{
	/** logger instance for this class */
	private static final Log log = LogFactory.getLog(InfomanLiteEventParser.class);

	private static final int FRAME_TYPE_BASIC = 0x01;

	private static final int FRAME_TYPE_DEVICE_START_WAKEUP = 0x02;

	private static final int FRAME_TYPE_DEVICE_START_WAKEUP_WITH_DEV_INFO = 0x03;

	private static final int FRAME_TYPE_MONITORING = 0x04;

	private static final int FRAME_TYPE_DEBUG_MSG_INCLUDING_BASIC_FRAME = 0x05;

	private static final int FRAME_TYPE_DEBUG_MSG_EXCLUDING_BASIC_FRAME = 0x06;

	private static final int FRAME_TYPE_AUTHENTICATION_MESSAGE = 0x07;

	/**
	 * @param event the event send form the device
	 * @param parsedEventParameters map, should be filled with parsed key value information
	 * @param parsedMetaInfo map, should be filled with parsed event meta information
	 */
	public void parseEvent(Object event, Map parsedEventParameters, Map metaInfo)
	{
		try
		{
			InfomanLiteEvent infomanLiteEvent = (InfomanLiteEvent) event;

			metaInfo.put("hwsn", infomanLiteEvent.getSerialNumber());
			metaInfo.put("hwType", "infomanLite");

			final byte[] data = infomanLiteEvent.getData();

			ByteArrayLittleEndianParser dataParser = new ByteArrayLittleEndianParser(data);

			int frameType = getFrameType(dataParser);
			switch(frameType)
			{
				case(FRAME_TYPE_BASIC):
					parseBasicFrame(dataParser, parsedEventParameters, metaInfo);
					break;
				case(FRAME_TYPE_DEVICE_START_WAKEUP):
					parseDeviceStartWakeupFrame(dataParser, parsedEventParameters, metaInfo);
					break;
				case(FRAME_TYPE_DEVICE_START_WAKEUP_WITH_DEV_INFO):
					parseDeviceStartWakeupFrameWithDeviceInfo(dataParser, parsedEventParameters, metaInfo);
					break;
				case(FRAME_TYPE_MONITORING):
					parseMonitoringFrame(dataParser, parsedEventParameters, metaInfo);
					break;
				case(FRAME_TYPE_DEBUG_MSG_INCLUDING_BASIC_FRAME):
					parseDebugMsgInclBasicFrame(dataParser, parsedEventParameters, metaInfo);
					break;
				case(FRAME_TYPE_DEBUG_MSG_EXCLUDING_BASIC_FRAME):
					parseDebugMsgExclBasicFrame(dataParser, parsedEventParameters, metaInfo);
					break;
				case(FRAME_TYPE_AUTHENTICATION_MESSAGE):
					parseAuthenicationFrame(dataParser, parsedEventParameters, metaInfo);
					break;
				default:
					log.error("Unknown frame type: " + Integer.toHexString(frameType));
					break;
			}
		}
		catch(Exception ex)
		{
			log.error("runs into", ex);
		}
	}

	int getFrameType(ByteArrayLittleEndianParser dataParser)
	{
		return dataParser.getUnsignedByte(0);
	}

	/**
	 * Parses a basic frame and fills the event parameters and meta info Maps
	 * with the event values.
	 *
	 * @param dataParser the ByteArrayLittleEndianParser helper class
	 * @param parsedEventParameters the Map that should be filled with the event parameters
	 * @param metaInfo the Map that should be filled with meta info
	 *
	 * @throws IllegalArgumentException if some illegal values appear, e. g. timestamp millis are greater than 999 or below 0
	 */
	void parseBasicFrame(ByteArrayLittleEndianParser dataParser, Map parsedEventParameters, Map metaInfo) throws IllegalArgumentException
	{
		// basics
		parsedEventParameters.put("triggerReason", dataParser.getUnsignedByte(1));
		parsedEventParameters.put("timestamp", calculateTimestamp(dataParser.getUnsignedBytesAsLong(4, 4), dataParser.getUnsignedBits(2, 0, 10)));

		// position & movement
		
		// latitude and longitude according to the internal spc must be converted into rad milliseconds
		double angle = dataParser.getSignedBits(8, 0, 31); 
		// * 60 * 1000
		// / 100000
		angle /= 10;
		angle *= 6;
		parsedEventParameters.put("gps.latitude", (long) angle);
		
		
		angle = dataParser.getSignedBytes(12, 4); 
		// * 60 * 1000
		// / 100000
		angle /= 10;
		angle *= 6;
		parsedEventParameters.put("gps.longitude", (long) angle);

		float course = dataParser.getUnsignedBytes(16, 2);
		course /= 100;
		if(course > 360)
		{
			throw new IllegalArgumentException("Course is out of range: " + course);
		}
		parsedEventParameters.put("gps.course", course);
		
		float speed = dataParser.getUnsignedBits(18, 0, 18);
		speed /= 1000;		
		parsedEventParameters.put("gps.speed", speed);
		
		float altitude = dataParser.getSignedBits(20, 2, 18);
		altitude /= 10;
		parsedEventParameters.put("gps.altitude", altitude);
				
		parsedEventParameters.put("gps.distance", dataParser.getUnsignedBytesAsLong(23, 4));

		// position quality
		float dop = dataParser.getUnsignedBits(27, 0, 12);
		dop /= 100;
		parsedEventParameters.put("gps.hdop", dop);
		
		dop = dataParser.getUnsignedBits(28, 4, 12);
		dop /= 100;
		parsedEventParameters.put("gps.pdop", dop);				
		
		parsedEventParameters.put("gps.satellites", dataParser.getUnsignedBits(30, 0, 4));
		parsedEventParameters.put("gps.fixIndicator", dataParser.getUnsignedBits(30, 4, 3));
		parsedEventParameters.put("gps.trusted", dataParser.isSet(30, 7));

		// I/O
		float potential = dataParser.getUnsignedBits(31, 0, 12);
		potential /= 409.5f;
		parsedEventParameters.put("io.analog1", potential);
		
		potential = dataParser.getUnsignedBits(32, 4, 12);
		potential /= 409.5f;
		parsedEventParameters.put("io.analog2", potential);

		for(int i = 0; i < 5; i++)
		{
			parsedEventParameters.put("io.digitalIn" + i, dataParser.isSet(34, i));
		}

		for(int i = 0; i < 3; i++)
		{
			parsedEventParameters.put("io.digitalOut" + i, dataParser.isSet(34, i + 5));
		}

		parsedEventParameters.put("io.powerSupply", dataParser.isSet(35, 0));
		parsedEventParameters.put("io.batteryCondition", dataParser.getUnsignedBits(35, 1, 2));
		parsedEventParameters.put("io.batteryCharging", dataParser.isSet(35, 3));
		parsedEventParameters.put("io.gpsAntenna", dataParser.isSet(35, 4));
		parsedEventParameters.put("io.authStatus.auth", dataParser.isSet(35, 5));
		parsedEventParameters.put("io.authStatus.use", dataParser.isSet(35, 6));
		parsedEventParameters.put("io.moving", dataParser.isSet(35, 7));
		
		generateNMEARecords(parsedEventParameters);
	}
	
	void generateNMEARecords(Map parsedEventParameters)
	{
		Map<String, String> nmeaParams = new HashMap<String, String>();

		nmeaParams.put("time", Long.toString(parsedEventParameters.get("timestamp")));

		long latitude = parsedEventParameters.get("gps.latitude");
		nmeaParams.put("latitude", Long.toString(latitude));

		long longitude = parsedEventParameters.get("gps.longitude");
		nmeaParams.put("longitude", Long.toString(longitude));

		// speed
		NumberFormat numberFormat = NumberFormat.getInstance(Locale.US);
		numberFormat.setMaximumFractionDigits(2);
		numberFormat.setMinimumFractionDigits(1);
		nmeaParams.put("speed", numberFormat.format(parsedEventParameters.get("gps.speed")));

		// course
		nmeaParams.put("trackMadeGood", numberFormat.format(parsedEventParameters.get("gps.course")));

		// sats
		nmeaParams.put("sats", Integer.toString(parsedEventParameters.get("gps.satellites")));

		// hdop
		nmeaParams.put("hdop", numberFormat.format(parsedEventParameters.get("gps.hdop")));

		// altitude
		nmeaParams.put("altitude", numberFormat.format(parsedEventParameters.get("gps.altitude")));

		String rmc = GPSPosition.createNMEArecord("RMC", nmeaParams);
		String gga = GPSPosition.createNMEArecord("GGA", nmeaParams);

		parsedEventParameters.put("rmc", rmc);
		parsedEventParameters.put("gga", gga); 
	}

	/**
	 * Parses a device start/wakeup frame and fills the event parameters and meta info Maps
	 * with the event values.
	 *
	 * @param dataParser the ByteArrayLittleEndianParser helper class
	 * @param parsedEventParameters the Map that should be filled with the event parameters
	 * @param metaInfo the Map that should be filled with meta info
	 *
	 * @throws IllegalArgumentException if some illegal values appear
	 */
	void parseDeviceStartWakeupFrame(ByteArrayLittleEndianParser dataParser, Map parsedEventParameters, Map metaInfo)
	{
		// Basic frame
		parseBasicFrame(dataParser, parsedEventParameters, metaInfo);

		// Startup information
		parsedEventParameters.put("device.startStatus", dataParser.getUnsignedByte(36));
	}

	/**
	 * Parses a device start/wakeup frame with device information and fills the
	 * event parameters and meta info Maps with the event values.
	 *
	 * @param dataParser the ByteArrayLittleEndianParser helper class
	 * @param parsedEventParameters the Map that should be filled with the event parameters
	 * @param metaInfo the Map that should be filled with meta info
	 *
	 * @throws IllegalArgumentException if some illegal values appear
	 */
	void parseDeviceStartWakeupFrameWithDeviceInfo(ByteArrayLittleEndianParser dataParser, Map parsedEventParameters, Map metaInfo)
	{
		// Basic frame
		parseBasicFrame(dataParser, parsedEventParameters, metaInfo);

		// Startup information
		parseDeviceStartWakeupFrame(dataParser, parsedEventParameters, metaInfo);

		// Device information
		
		ByteArrayBigEndianParser bigEndian = new ByteArrayBigEndianParser(dataParser);		
		parsedEventParameters.put("device.productName", bigEndian.getString(37, 20));
		parsedEventParameters.put("device.swVersion", bigEndian.getString(57, 42));
		parsedEventParameters.put("device.hwVersion", bigEndian.getString(99, 50));
		parsedEventParameters.put("device.modemType", bigEndian.getString(149, 40));
		parsedEventParameters.put("device.modemManufacturer", bigEndian.getString(189, 15));
		parsedEventParameters.put("device.modemSwVersion", bigEndian.getString(204, 15));
		
		String iccid = bigEndian.getBCD(219, 0, 20);
		iccid = iccid.replace('F', ' ').trim();
		iccid = iccid.replace('f', ' ').trim();
		parsedEventParameters.put("device.iccid", iccid);
		
		parsedEventParameters.put("device.imsi", bigEndian.getBCD(229, 0, 15));
		parsedEventParameters.put("device.vehicleID", convertByteArrayToString(bigEndian.getByteArray(237, 0, 64)));
	}
	
	String convertByteArrayToString(Object obj)
	{
		String str;	
		if(obj instanceof byte[])
		{
			str = ByteUtil.toHexString((byte[]) obj);
			if(str.contains(" "))
			{   
				str = str.replaceAll(" ", "");
			}					
		}
		else
		{
			str = obj.toString();
		}

		return str;
	}

	/**
	 * Parses a monitoring frame and fills the event parameters and meta
	 * info Maps with the event values.
	 *
	 * @param dataParser the ByteArrayLittleEndianParser helper class
	 * @param parsedEventParameters the Map that should be filled with the event parameters
	 * @param metaInfo the Map that should be filled with meta info
	 *
	 * @throws IllegalArgumentException if some illegal values appear
	 */
	void parseMonitoringFrame(ByteArrayLittleEndianParser dataParser, Map parsedEventParameters, Map metaInfo)
	{
		// Basic frame
		parseBasicFrame(dataParser, parsedEventParameters, metaInfo);

		// Monitoring
		parsedEventParameters.put("io.temp", dataParser.getSignedBytes(36,1));
		parsedEventParameters.put("monitoring.csq.rssi", dataParser.getUnsignedBits(37, 0, 6));
		parsedEventParameters.put("monitoring.csq.ber", dataParser.getUnsignedBits(38, 0, 4));
		parsedEventParameters.put("monitoring.cregStatus", dataParser.getUnsignedBits(38, 4, 3));
		parsedEventParameters.put("monitoring.lac",  dataParser.getUnsignedBytes(39, 2));
		parsedEventParameters.put("monitoring.cid", dataParser.getUnsignedBytes(41, 2));
		parsedEventParameters.put("monitoring.imsi", dataParser.getBCD(43, 0, 15));
		parsedEventParameters.put("monitoring.gpsAntenna.powerSupply", dataParser.getUnsignedBits(51, 0, 12));
		parsedEventParameters.put("monitoring.gpsAntenna.absence", dataParser.isSet(52, 4));
		parsedEventParameters.put("monitoring.gpsAntenna.shortCircuite", dataParser.isSet(52, 5));
		parsedEventParameters.put("monitoring.powerSupply.voltage", dataParser.getUnsignedBits(53, 0, 12));
	}

	/**
	 * Parses a debug message with included basic frame and fills the event
	 * parameters and meta info Maps with the event values.
	 *
	 * @param dataParser the ByteArrayLittleEndianParser helper class
	 * @param parsedEventParameters the Map that should be filled with the event parameters
	 * @param metaInfo the Map that should be filled with meta info
	 *
	 * @throws IllegalArgumentException if some illegal values appear
	 */
	void parseDebugMsgInclBasicFrame(ByteArrayLittleEndianParser dataParser, Map parsedEventParameters, Map metaInfo)
	{
		// Basic frame
		parseBasicFrame(dataParser, parsedEventParameters, metaInfo);

		// Debug message
		ByteArrayBigEndianParser bigEndian = new ByteArrayBigEndianParser(dataParser);		
		parsedEventParameters.put("debug.msgType", dataParser.getUnsignedByte(36));
		parsedEventParameters.put("debug.msg", bigEndian.getString(37, 40));
	}

	/**
	 * Parses a plain debug message without a basic frame and fills the
	 * event parameters and meta info Maps with the event values.
	 *
	 * @param dataParser the ByteArrayLittleEndianParser helper class
	 * @param parsedEventParameters the Map that should be filled with the event parameters
	 * @param metaInfo the Map that should be filled with meta info
	 *
	 * @throws IllegalArgumentException if some illegal values appear
	 */
	void parseDebugMsgExclBasicFrame(ByteArrayLittleEndianParser dataParser, Map parsedEventParameters, Map metaInfo)
	{
		// timestamp
		parsedEventParameters.put("timestamp", calculateTimestamp(dataParser.getUnsignedBytesAsLong(3, 4), dataParser.getUnsignedBits(1, 0, 10)));

		// Debug message
		ByteArrayBigEndianParser bigEndian = new ByteArrayBigEndianParser(dataParser);		
		parsedEventParameters.put("debug.msgType", dataParser.getUnsignedByte(7));
		parsedEventParameters.put("debug.msg", bigEndian.getString(8, 40));
	}

	/**
	 * Parses a debug message with included basic frame and fills the event
	 * parameters and meta info Maps with the event values.
	 *
	 * @param dataParser the ByteArrayLittleEndianParser helper class
	 * @param parsedEventParameters the Map that should be filled with the event parameters
	 * @param metaInfo the Map that should be filled with meta info
	 *
	 * @throws IllegalArgumentException if some illegal values appear
	 */
	void parseAuthenicationFrame(ByteArrayLittleEndianParser dataParser, Map parsedEventParameters, Map metaInfo)
	{
		// Basic frame
		parseBasicFrame(dataParser, parsedEventParameters, metaInfo);

		// User ID
		ByteArrayBigEndianParser bigEndian = new ByteArrayBigEndianParser(dataParser);		
		parsedEventParameters.put("auth.userID", convertByteArrayToString(bigEndian.getByteArray(36, 0, 64)));
	}

	/**
	 * Calculates an event timestamp and returns it in a java milliseconds
	 * timestamp
	 *
	 * @params seconds seconds since 01.01.1970
	 * @params millis additional milliseconds (0 - 999)
	 *
	 * @return java milliseconds timestamp
	 *
	 * @throws IllegalArgumentException if millis are less than 0 or greater than 999
	 */
	long calculateTimestamp(final long seconds, final int millis) throws IllegalArgumentException
	{
		if(millis < 0 || millis > 999)
		{
			throw new IllegalArgumentException("Milliseconds out of range: " + millis);
		}

		return (seconds * 1000) + millis;
	}
}

