package infomanLite;

import de.proveo.wwt.logic.ejb.dataIn.event.mapper.*;

import de.proveo.event.util.efm.EFMEventUtil;
import de.proveo.util.geo.GPSPosition;

import de.proveo.eventbase.EventConstants;

import de.proveo.wwt.logic.ejb.state.CurrentStateFacadeLocal;
import de.proveo.wwt.logic.ejb.state.CurrentStateFacadeStruct;
import de.proveo.wwt.datamodel.measurement.MeasurementCache;

import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Iterator;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.ArrayList;
import java.util.HashMap;

import org.apache.commons.logging.Log;

/**
 * Abstract Script for: Liquid Level Measurement based on analog values.
 * 
 * The script is compatible only with and above 4.6.0!
 * 
 * Should be used for entry point <code>infoman.event.3rdparty.eventMapper.lib</code>
 * 
 * @author <a href="mailto:jbader@proveo.com">Joachim Bader</a>
 * $Rev: 21140 $
 */
class BaseInfomanLiteLiquidLevelMapping implements EventMapper
{
	/** format helper instance to get two three measurement values */
	private static final MessageFormat twoDigitsFormat = new MessageFormat("{0,number,#0.000}", Locale.US);

	/** logger instance */
	private final Log log;
	
	protected BaseInfomanLiteLiquidLevelMapping(Log log)
	{
		this.log = log;
	}
	
	/**
	 * @return the threshold which must be exceeded to fire a new event
	 * a negative value or zero is disabling the threshold feature
	 */
	float getMeasurementEventThreshold()
	{
		return 0.0;
	}	
	
	/** 
	 * Return the analog value for conversion.
	 * Should be overwritten by the conrect script. 
	 **/
	Float getAnalogValue(Map eventParameters)
	{
		return null;
	}
	
	/**
	 * Optional state model if for volume driven state models.
	 * Should be overwritten by the conrect script. 
	 */
	Integer getVolumeStatusModel()
	{
		return null;
	}
	
	/**
	 * @return the measurment event type to use for the generated measurement events
     * returning <code>null</code> is disabling the event
	 */
	Integer getMeasurementEventType()
	{
		return null;
	}

	/**
     * @return the raw value list
     */
	ArrayList<Float> getRawValues()
	{
		return null;
	}

	/**
     * @return the corresponding volume list
     * @see #getRawValues()
     */
	ArrayList<Float> getVolumeValues()
	{
		return null;
	}
	
	Float getMaxVolume()
	{
		Float maxVolume = null;
		ArrayList volumeValues = getVolumeValues();
		if(volumeValues != null)
		{
			maxVolume = volumeValues.get(volumeValues.size()-1);
		}

		return maxVolume;
	}
		

	Float calculateVolume(float raw)
	{
		if(raw==null)
		{
			return null;
		}

		if((getRawValues()==null) || (getVolumeValues()==null))
		{
			return null;
		}

		if(getRawValues().size()!=getVolumeValues().size())
		{
			throw new Exception("raw value and volume table doesn't match");
		}

		if(raw<=0)
		{
			return 0.0;
		}

		for(int i=1; i<=getRawValues().size()-1; i++)
		{
			float value1=getRawValues()[i];
			
			if(raw <= value1) 
			{
				Float volume=getVolumeValues()[i-1];
				int litersBase=getVolumeValues()[i]-getVolumeValues()[i-1];
				
				int value2=getVolumeValues()[i-1];
				
				Float factor=(value1-value2)/litersBase;
				volume+=(raw-value2)/factor;
				
				return volume;
			}
		}		
		
		return null;
	}
		
	boolean isUpdateNecessary(Float newValue, List<Long> efmUnitIds, FacadeWrapper facadeWrapper)
	{
		if(getMeasurementEventType()==null)
		{
			return false;
		}

		if(getMeasurementEventThreshold()<0)
		{
			return true;
		}		
		
		MeasurementCache measurementCache = facadeWrapper.getMeasurementFacade().getMeasurementCacheByUnitAndType(efmUnitIds.get(0), getMeasurementEventType());
		if(measurementCache==null)
		{
			return true;
		}
		
		float diff = Math.abs(measurementCache.getValue()-newValue.floatValue());
		
		return diff>getMeasurementEventThreshold();		
	}
	
    public List createEFMevents(Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{		
		List<Properties> efmEvents = new ArrayList<Properties>();
		
		if(!eventParameters.isEmpty())
		{

			Float analogValue = getAnalogValue(eventParameters);
			if(analogValue!=null)
			{
				List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");
				
				Float volume = calculateVolume(analogValue);

				if((volume!=null) && (isUpdateNecessary(volume, efmUnitIds, facadeWrapper)))
				{
					if(getMeasurementEventType()!=null)
					{					
						Properties event = new Properties();
						EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get("timestamp"));
						event.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(getMeasurementEventType()));

						Object[] objArr = new Object[1];
						objArr[0] = volume;
						String mValueStr = twoDigitsFormat.format(objArr);					
						event.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValueStr);

						efmEvents.add(event);
					}
					
					if((getVolumeStatusModel()!=null) && (getMaxVolume()!=null))
					{
						float volumePercentage = volume.floatValue()/getMaxVolume().floatValue();

						
						Integer stateId = getVolumeStatus(volumePercentage);
						
						if(stateId!=null)
						{
							Properties event = new Properties();
							EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get("timestamp"));
							event.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, Integer.toString(getVolumeStatusModel()));
							event.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(stateId));

							efmEvents.add(event);
						}
					}
				}
			}
		}
		
		return efmEvents;
	}
		
	Integer getVolumeStatus(Float percentage)
	{
		int volumeStatus;
		if(percentage<0.10)
		{
			volumeStatus = getVolumeStatusModel();
		}
		else if(percentage<0.20)
		{
			volumeStatus = getVolumeStatusModel()+1;
		}
		else if(percentage<0.30)
		{
			volumeStatus = getVolumeStatusModel()+2;
		}
		else if(percentage<0.40)
		{
			volumeStatus = getVolumeStatusModel()+3;
		}
		else if(percentage<0.50)
		{
			volumeStatus = getVolumeStatusModel()+4;
		}
		else if(percentage<0.60)
		{
			volumeStatus = getVolumeStatusModel()+5;
		}
		else if(percentage<0.70)
		{
			volumeStatus = getVolumeStatusModel()+6;
		}
		else if(percentage<0.80)
		{
			volumeStatus = getVolumeStatusModel()+7;
		}
		else if(percentage<0.90)
		{
			volumeStatus = getVolumeStatusModel()+8;
		}
		else // 90-100%
		{
			volumeStatus = getVolumeStatusModel()+9;
		}				

		return volumeStatus;
	}	
}
