package infomanLite;

import de.proveo.wwt.logic.ejb.dataIn.event.mapper.*;

import de.proveo.event.util.efm.EFMEventUtil;
import de.proveo.util.geo.GPSPosition;

import de.proveo.eventbase.EventConstants;

import java.util.*;

import java.text.*;

import de.proveo.event.util.raw.ByteUtil;
import de.proveo.wwt.logic.ejb.state.CurrentStateFacadeStruct;
import de.proveo.wwt.datamodel.measurement.MeasurementCache;

/**
 * Base Event Mapper Scripts for InfomanLite
 *
 * Should be used for entry point <code>infoman.event.3rdparty.eventMapper.lib</code>
 *
 * Generate events:
 * <ul>
 *   <li>Position with distance measurement</li>
 *   <li>Motorstatus: Engine on/off</li>
 *   <li>Operationstatus: available, work and drive mode</li>
 * </ul>
 * 
 * Methods which sould be ovwritten by a concret script:
 * Boolean isEngineOnCondition(Map eventParameters)
 * Boolean isWorkCondition(Map eventParameters)
 * Boolean isDriveCondition(Map eventParameters)
 * 
 * Integer getDistanceMeasurementType()
 *
 * @author <a href="mailto:jbader@proveo.com">Joachim Bader</a>
 * $Rev: 22155 $
 */
class BaseInfomanLiteEventMapper implements EventMapper 
{
	
	// proveo's standard motor status model
	protected static final int MOTOR_STATE_MODEL_ID = 2010;
	protected static final int MOTOR_OFF_STATE_ID = 2011;
	protected static final int MOTOR_ON_STATE_ID = 2012;
	protected static final int MOTOR_RUNS_UNNECESSARY_STATE_ID = 2013;
	
	// proveo's standard operation status model
	protected static final int OPERATION_STATE_MODEL_ID = 2050;
	protected static final int OPERATION_AVAILABLE_STATE_ID = 2054;
	protected static final int OPERATION_WORK_STATE_ID = 2053;
	protected static final int OPERATION_DRIVE_STATE_ID = 2052;
	protected static final int OPERATION_STANDBY_STATE_ID = 2055;
	
	
	Boolean isEngineOnCondition(Map eventParameters)
	{
		// no egnine condition defined
		return null;
	}
	
	Boolean isWorkCondition(Map eventParameters)
	{
		// no work condition defined
		return null;
	}
	
	Boolean isDriveCondition(Map eventParameters)
	{
		// no drive condition defined
		return null;
	}
	
	Integer getDistanceMeasurementType()
	{
		// no distance measurement type defined
		return null;
	}
	
	Integer getDistanceMeasruementUpdateThreshold()
	{
		return 10000;
	}
	
	public List createEFMevents(Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{
        List<Properties> efmEvents = new ArrayList<Properties>();
		
		if(!eventParameters.isEmpty())
		{
			Properties geoEvent = createPositionEvent(efmEvents, eventParameters, metainfo, facadeWrapper);

			createStatusModelEvents(efmEvents, eventParameters, metainfo, facadeWrapper);
		}
		
		return efmEvents;
	}
	
	/**
	 * @return new state or null if no update is necessary
	 */
	Integer evaluateEngineStatusModel(CurrentStateFacadeStruct currentMotorState, Boolean motorOn)
	{
		// we cannot determine anything
		if(motorOn==null)
		{
			return null;
		}
		
		// status modell is not initialized yet
		if(currentMotorState==null)
		{				
			if(motorOn)
			{
				return MOTOR_ON_STATE_ID;
			}
			else
			{
				return MOTOR_OFF_STATE_ID;
			}
		}
		
		switch(currentMotorState.getStateId())
		{
			case MOTOR_OFF_STATE_ID:
				if(motorOn)
				{
					return MOTOR_ON_STATE_ID;
				}
				break;
			
			case MOTOR_ON_STATE_ID:
			case MOTOR_RUNS_UNNECESSARY_STATE_ID:
				if(!motorOn)
				{
					return MOTOR_OFF_STATE_ID;
				}
				break;
		}
		
		return null;
	}
	
	/**
	 * @return new state or null if no update is necessary
	 */
	Integer evaluateOpeationStatusModel(CurrentStateFacadeStruct currentOperationState, Boolean workCondition, Boolean driveCondition)
	{
		// we cannot determine anything
		if((workCondition==null) && (driveCondition==null))
		{
			return null;
		}
		else if(workCondition==null)
		{
			// no work condition defined, the vehicle will never work
			workCondition = Boolean.FALSE;			
		}
		else if(driveCondition==null)
		{
			// no drive condition defined, the vehicle will never drive
			driveCondition = Boolean.FALSE;
		}
		
		// status modell is not initialized yet
		if(currentOperationState==null)
		{				
			if(workCondition)
			{
				return OPERATION_WORK_STATE_ID;
			}
			else if(driveCondition)
			{
				return OPERATION_DRIVE_STATE_ID;
			}
			else
			{
				return OPERATION_AVAILABLE_STATE_ID;
			}
		}
		
		
		switch(currentOperationState.getStateId())
		{
			case OPERATION_AVAILABLE_STATE_ID:			
			case OPERATION_STANDBY_STATE_ID:
				if(workCondition)
				{
					return OPERATION_WORK_STATE_ID;
				}
				else if(driveCondition)
				{
					return OPERATION_DRIVE_STATE_ID;
				}
				break;
			
			case OPERATION_WORK_STATE_ID:
				if((!workCondition) && driveCondition)
				{
					return OPERATION_DRIVE_STATE_ID;
				}
				else if((!workCondition) && (!driveCondition))
				{
					return OPERATION_AVAILABLE_STATE_ID;
				}
				break;
				
			case OPERATION_DRIVE_STATE_ID:	
				if(workCondition)
				{
					return OPERATION_WORK_STATE_ID;
				}
				else if((!workCondition) && (!driveCondition))
				{
					return OPERATION_AVAILABLE_STATE_ID;
				}
				break;			
		}
		
		return null;
	}
	
	
	void createStatusModelEvents(List<Properties> efmEvents, Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{
		List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");
		
		Boolean motorOn = isEngineOnCondition(eventParameters);
		CurrentStateFacadeStruct currentState = facadeWrapper.getCurrentStateFacade().getCurrentState(efmUnitIds.get(0), MOTOR_STATE_MODEL_ID);
		Integer newState = evaluateEngineStatusModel(currentState, motorOn);
		
		if(newState!=null)
		{
			Properties event = new Properties();
			EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get("timestamp"));
			event.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, Integer.toString(MOTOR_STATE_MODEL_ID));
			event.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(newState));			
			
			efmEvents.add(event);
		}
		
		Boolean workCondition = isWorkCondition(eventParameters);
		Boolean driveCondition = isDriveCondition(eventParameters);
		currentState = facadeWrapper.getCurrentStateFacade().getCurrentState(efmUnitIds.get(0), OPERATION_STATE_MODEL_ID);
		newState = evaluateOpeationStatusModel(currentState, workCondition, driveCondition);
		
		if(newState!=null)
		{
			Properties event = new Properties();
			EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get("timestamp"));
			event.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, Integer.toString(OPERATION_STATE_MODEL_ID));
			event.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(newState));			
			
			efmEvents.add(event);
		}		
	}	
	
	Properties createPositionEvent(List<Properties> efmEvents, Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{
		Properties geoEvent = null;

		// Position Event
		if(eventParameters.containsKey("rmc"))
		{
			List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");	

			geoEvent = new Properties();
			EFMEventUtil.addGeneralParameters(geoEvent, efmUnitIds, (Long) eventParameters.get("timestamp"));
			
			// todo: all position are trusted right now
			geoEvent.setProperty(EventConstants.ATTRIBUTE_GEO_TRUSTED, "1");
			
			geoEvent.setProperty(EventConstants.ATTRIBUTE_GEO_RMC, (String) eventParameters.get("rmc"));
			geoEvent.setProperty(EventConstants.ATTRIBUTE_GEO_GGA, (String) eventParameters.get("gga"));
			geoEvent.setProperty(EventConstants.ATTRIBUTE_GEO_REASON, Integer.toString(eventParameters.get("triggerReason")));

			if((getDistanceMeasurementType()!=null) && eventParameters.containsKey("gps.distance") && isDistanceUpdateNecessary(efmUnitIds, eventParameters.get("gps.distance"), facadeWrapper))
			{
				geoEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(getDistanceMeasurementType()));
				geoEvent.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, Long.toString(eventParameters.get("gps.distance")));
			}

			efmEvents.add(geoEvent);			
		}
		
		return geoEvent;
	}	
	
	boolean isDistanceUpdateNecessary(List<Long> efmUnitIds, Long distance, FacadeWrapper facadeWrapper)
	{
		if(distance==null)
		{
			return false;
		}
		
		Integer threshold = getDistanceMeasruementUpdateThreshold();
		if(threshold==null)
		{
			return true;
		}
		
		MeasurementCache measurementCache = facadeWrapper.getMeasurementFacade().getMeasurementCacheByUnitAndType(efmUnitIds.get(0), getDistanceMeasurementType());
		if(measurementCache==null)
		{
			return true;
		}
		
		long diff = Math.abs(measurementCache.getValue()-distance.longValue());
		
		return diff>threshold.intValue();		
	}

}