package tkchel;

import de.proveo.wwt.logic.ejb.dataIn.event.mapper.*;

import de.proveo.event.util.efm.EFMEventUtil;
import de.proveo.util.geo.GPSPosition;

import de.proveo.eventbase.EventConstants;

import de.proveo.wwt.logic.ejb.state.CurrentStateFacadeLocal;
import de.proveo.wwt.logic.ejb.state.CurrentStateFacadeStruct;
import de.proveo.wwt.datamodel.measurement.MeasurementCache;

import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Iterator;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.ArrayList;
import java.util.HashMap;

import org.apache.commons.logging.Log;

/**
 * Abstract Script for: Fuel Measurement by Liquid Level Sensor measurements.
 * Use LLS1 and LLS2.
 * 
 * The script is compatible only with and above 4.6.0!
 * 
 * Should be used for entry point <code>infoman.event.3rdparty.eventMapper.lib</code>
 * 
 * A direved script for a specific unit should implement:
 * <code>
 * ArrayList getLLS1Values()   // a list of raw values
 * ArrayList getLLS1Liters()   // a list of liters corresponding to the above defined raw values
 * int getMeasurementEventType() // the measurement type to use
 * float getLiterEventThreshold() // minimum liter change betwenn two events
 * 
 * public TKChelLLSFuelLevelMappingSVO()
 * {
 * 	super(LogFactory.getLog(TKChelLLSFuelLevelMappingSVO.class));
 * }
 * </code>
 * 
 * @author <a href="mailto:jbader@proveo.com">Joachim Bader</a>
 * $Rev: 21140 $
 */
class BaseTKChelLLSFuelLevelMappingSVO implements EventMapper
{
	private final Log log;
	
	public BaseTKChelLLSFuelLevelMappingSVO(Log log)
	{
		this.log = log;
	}
	
	/**
	 * @return the threshold which must be exceeded to fire a new event
	 * a negative is disabling the threshold feature
	 */
	float getLiterEventThreshold()
	{
		return 0.0;
	}	
	
	/**
	 * @return the measurment event type to use for the generated measurement events
	 */
	int getMeasurementEventType()
	{
		return 310;
	}
	
	Float getMaxLiters()
	{
		float maxLiters = null;
		ArrayList lLS1Liters = getLLS1Liters();
		ArrayList lLS2Liters = getLLS2Liters();
		if(lLS1Liters != null){
			if(lLS2Liters != null){
				maxLiters = lLS1Liters.get(lLS1Liters.size()-1)+lLS2Liters.get(lLS2Liters.size()-1);
			}
			else{
				maxLiters = lLS1Liters.get(lLS1Liters.size()-1);
			}
		}
		else{
			if(lLS2Liters != null){
				maxLiters = lLS2Liters.get(lLS2Liters.size()-1)
			}
		}
		return maxLiters;
	}
	
	Integer getExpectedRecordType()
	{
		return new Integer(8);
	}
	
	ArrayList getLLS1Values()
	{
		return null;
	}
	
	ArrayList getLLS1Liters()
	{
		return null;
	}
	
	ArrayList getLLS2Values()
	{
		return null;
	}
	
	ArrayList getLLS2Liters()
	{
		return null;
	}
	
	/**
	 * Fuel level calculation 
	 * @return return the fuel level in liters; null if you don't want any event
	 */
	Float calculateLiquidLevelLiterLLS(Integer lls1, Boolean lls1Valid, Integer lls2, Boolean lls2Valid)
	{
		if(log.isTraceEnabled())
		{
			log.trace("calculateLiquidLevelLiterLLS() "+lls1+" "+lls2);
		}		
				
		Float lls1Liters = getLLSLiters(lls1, lls1Valid, getLLS1Values(), getLLS1Liters());
		Float lls2Liters = getLLSLiters(lls2, lls2Valid, getLLS2Values(), getLLS2Liters());

		Float total = lls1Liters;
		if(total==null)
		{
			total = lls2Liters;
		}
		else if(lls2Liters!=null)
		{
			total += lls2Liters;
		}
		
		if(log.isTraceEnabled())
		{
			log.trace("calculateLiquidLevelLiterLLS() "+lls1+" "+lls2+" Liters: "+lls1Liters+" "+lls2Liters +" Total: " + total);
		}		
		
		return total;
	}	
	
	Float getLLSLiters(Integer lls, Boolean llsValid, ArrayList llsValues, ArrayList llsLiters)
	{
		if(!Boolean.TRUE.equals(llsValid))
		{
			return null;
		}
		
		if((llsValues==null) || (llsLiters==null))
		{
			return null;
		}
		
		if(llsValues.size()!=llsLiters.size())
		{
			throw new Exception("sensor table and liters table doesn't have the wrong size");
		}
		
		if(lls<=0) return 0.0;

		for(int i=1; i<=llsValues.size()-1; i++)
		{
			int value1=llsValues[i];
			
			if(lls <= value1) 
			{
				Float liters=llsLiters[i-1];
				int litersBase=llsLiters[i]-llsLiters[i-1];
				
				int value2=llsValues[i-1];
				
				Float factor=(value1-value2)/litersBase;
				liters+=(lls-value2)/factor;
				
				return liters;
			}
		}		
		
		return null;		
	}
	
	private static final MessageFormat twoDigitsFormat = new MessageFormat("{0,number,#0.00}", Locale.US);
	
	/**
	 * @return the fuel level in percentage (0.0-1.0); return null if you don't want a fuel level status event
	 */
	Float calculateFuelLevelPercentage(float fuelLiters)
	{
		return fuelLiters/getMaxLiters();
	}	
	
	boolean isUpdateNecessary(Float newValue, List<Long> efmUnitIds, FacadeWrapper facadeWrapper)
	{
		if(getLiterEventThreshold()<0)
		{
			return true;
		}		
		
		MeasurementCache measurementCache = facadeWrapper.getMeasurementFacade().getMeasurementCacheByUnitAndType(efmUnitIds.get(0), getMeasurementEventType());
		if(measurementCache==null)
		{
			return true;
		}
		
		float diff = Math.abs(measurementCache.getValue()-newValue.floatValue());
		
		return diff>getLiterEventThreshold();		
	}
	
    public List createEFMevents(Map eventParameters, Map metainfo, FacadeWrapper facadeWrapper)
	{		
		final List efmEvents = new ArrayList();
		final List<Long> efmUnitIds = (List<Long>) metainfo.get("efm.unitIds");		
		
		if(efmUnitIds==null)
		{
			return null;
		}
		
		int recordIndex = 0;
		boolean recordAvailable = true;
		while(recordAvailable)
		{
			recordAvailable = eventParameters.containsKey(recordIndex+".timestamp");
			if(recordAvailable)
			{
				Integer recordType = (Integer) eventParameters.get(recordIndex+".tkchel.recordType");
				if(getExpectedRecordType().equals(recordType))
				{				
					Integer lls1 = getLLS(eventParameters, recordIndex, 1);
					Integer lls2 = getLLS(eventParameters, recordIndex, 2);

					Boolean lls1Valid = (Boolean) eventParameters.get(recordIndex+".lls1.valid");
					Boolean lls2Valid = (Boolean) eventParameters.get(recordIndex+".lls2.valid");
					
					Float liquidLevelLiters = calculateLiquidLevelLiterLLS(lls1, lls1Valid, lls2, lls2Valid);
					
					if(log.isTraceEnabled())
					{
						log.trace("createEFMevents() "+liquidLevelLiters+" liters");
					}
					
					if(isUpdateNecessary(liquidLevelLiters, efmUnitIds, facadeWrapper))
					{
						Properties event = createEvent(efmUnitIds, eventParameters, recordIndex, getMeasurementEventType(), liquidLevelLiters);				
						if(event!=null)
						{
							efmEvents.add(event);
						}

						if(liquidLevelLiters!=null)
						{
							Float fuelLevelPercentage = calculateFuelLevelPercentage(liquidLevelLiters);

							if(log.isTraceEnabled())
							{
								log.trace("createEFMevents() "+fuelLevelPercentage+" %");
							}


							Properties eventFuelLevelStatus = createEventFuelLevelStatus(efmUnitIds, eventParameters, recordIndex, fuelLevelPercentage);				
							if(eventFuelLevelStatus!=null)
							{
								efmEvents.add(eventFuelLevelStatus);
							}
						}
					}
				}
				
				recordIndex++;
			}					
		}
		
		return efmEvents;
	}
	
	Properties createEventFuelLevelStatus(List<Long> efmUnitIds, Map eventParameters, int recordIndex, Float liquidLevelPercentage)
	{
		if(liquidLevelPercentage==null)
		{
			return null;
		}
		
		Properties event = new Properties();
		EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get(recordIndex+".timestamp"));
		event.setProperty(EventConstants.ATTRIBUTE_STATEMODEL, "2030");
		event.setProperty(EventConstants.ATTRIBUTE_STATEID, Integer.toString(getFuelStatus(liquidLevelPercentage)));

		return event;		
	}
	
	int getFuelStatus(Float percentage)
	{
		int fuelStatus;
		if(percentage<0.10)
		{
			fuelStatus = 2030;
		}
		else if(percentage<0.20)
		{
			fuelStatus = 2031;
		}
		else if(percentage<0.30)
		{
			fuelStatus = 2032;
		}
		else if(percentage<0.40)
		{
			fuelStatus = 2033;
		}
		else if(percentage<0.50)
		{
			fuelStatus = 2034;
		}
		else if(percentage<0.60)
		{
			fuelStatus = 2035;
		}
		else if(percentage<0.70)
		{
			fuelStatus = 2036;
		}
		else if(percentage<0.80)
		{
			fuelStatus = 2037;
		}
		else if(percentage<0.90)
		{
			fuelStatus = 2038;
		}
		else // 90-100%
		{
			fuelStatus = 2039;
		}				

		return fuelStatus;
	}
	
	Integer getLLS(Map eventParameters, int recordIndex, int number)
	{
		return eventParameters.get(recordIndex+".lls"+number);
	}
	
	Properties createEvent(List<Long> efmUnitIds, Map eventParameters, int recordIndex, int measurementType, Float liquidLevelLiters)
	{
		if(liquidLevelLiters==null)
		{
			return null;
		}
		
		Properties event = new Properties();
		EFMEventUtil.addGeneralParameters(event, efmUnitIds, (Long) eventParameters.get(recordIndex+".timestamp"));
		event.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_TYPE, Integer.toString(measurementType));
		Object[] objArr = new Object[1];
		objArr[0] = liquidLevelLiters;
		String mValueStr = twoDigitsFormat.format(objArr);
		event.setProperty(EventConstants.ATTRIBUTE_MEASUREMENT_VALUE, mValueStr);

		return event;
	}	
}
