package telic

import de.proveo.event.telic.model.*;
import de.proveo.event.telic.parser.*;
import de.proveo.util.geo.GPSPosition;

import de.proveo.event.util.efm.groovy.EventParser;
import java.util.*;

import org.apache.commons.lang.StringUtils;

/**
 *
 * @author <a href="mailto:jbader@zebra.com">Joachim Bader</a>
 * $Rev: 22427 $
 */
class TelicEventParser implements EventParser 
{
    private static final String IMEI_PREFIX = "357023007";
	
	/**
     * 
     * @param event the event send form the device
     * @param parsedEventParameters map, should be filled with parsed key value information
     * @param parsedMetaInfo map, should be filled with parsed event meta information
     */
    public void parseEvent(Object event, Map parsedEventParameters, Map metaInfo)
    {
		try
		{
            TelicEvent telicEvent = (TelicEvent) event;
			
			metaInfo.put("hwsn", IMEI_PREFIX+telicEvent.getIdSequence().getPartOfIMEI());
			metaInfo.put("hwType", "telic");			
            
            final int msgCodeContentPart = (telicEvent.getMsgCode() & 0xF0) >> 4;

            switch(msgCodeContentPart)
            {
                case 0: // acknowledge
                    break;

                case 1: // basic configuration
                    break;
                case 2: // logging data
                    parseLoggingDataEvent(telicEvent, parsedEventParameters, metaInfo);
                    break;
                case 3: // event data
                    parseEventDataEvent(telicEvent, parsedEventParameters, metaInfo);
                    break;
                case 4: // I/O configuration, status response
                    break;

                case 5: // reserved
                case 6: // reserved
                    break;

                case 7: // GSM network information
                    break;

                case 8: // reserved
                case 9: // reserved
                    break;

                case 0xA: // geofencing
                    break;
                case 0xB: // IP parameter configuration
                    break;
                case 0xC: // expert configuration
                    break;

                case 0xD: // reserved
                case 0xE: // reserved
                case 0xF: // reserved
                    break;

                default:
                    throw new IllegalArgumentException("unkown msgCodeContentPart: "+msgCodeContentPart);
            }            
        }
		catch(Exception ex)
		{
			ex.printStackTrace();
		}        
    }
    
    void parseLoggingDataEvent(TelicEvent event, Map parsedEventParameters, Map metaInfo)
    {
        parseEventDataEvent(event, parsedEventParameters, metaInfo);
    }
    
    void parseEventDataEvent(TelicEvent telicEvent, Map parsedEventParameters, Map metaInfo)
    {
        if(telicEvent.getData()==null)
        {
            // it's only a keep alive event
            return;
        }
        
        final String[] tokens = StringUtils.splitByWholeSeparatorPreserveAllTokens(telicEvent.getData(), ",");
        
        String eventCause = tokens[0];
        String eventTimestamp = tokens[1];
        String eventInfo = tokens[2];
        parsePositionEvent(parsedEventParameters, tokens);
        String milage = tokens[13];
        String mccMNC = tokens[14];
        String digitalInput = tokens[15];
        String digitalOutput = tokens[16];
        String analogueInput1 = tokens[17];
        String analogueInput2 = tokens[18];


        //parsedEventParameters.put("idSequence", idSequence);
        parsedEventParameters.put("telicEvent", telicEvent);
        parsedEventParameters.put("eventCause", eventCause);
        parsedEventParameters.put("timestamp", TelicParserUtil.getTimestamp(eventTimestamp));
        parsedEventParameters.put("eventInfo", eventInfo);
        parsedEventParameters.put("milage", milage);
        parsedEventParameters.put("mccMNC", mccMNC);

        parsedEventParameters.put("in1", digitalInput.substring(0, 1).equals("1"));
        parsedEventParameters.put("in2", digitalInput.substring(1, 2).equals("1"));

        parsedEventParameters.put("out1", digitalOutput.substring(0, 1).equals("1"));
        parsedEventParameters.put("out2", digitalOutput.substring(1, 2).equals("1"));

        parsedEventParameters.put("an1", Integer.parseInt(analogueInput1));
        parsedEventParameters.put("an2", Integer.parseInt(analogueInput2));
        
    }
 

	void parsePositionEvent(Map eventParameters, String[] tokens)
	{
		Map<String, String> parameters = new HashMap<String, String>();

		String gpsTimestamp = tokens[3];
		long timestamp = TelicParserUtil.getTimestamp(gpsTimestamp);
		parameters.put("time", Long.toString(timestamp));

		String logitude = tokens[4];
		long msec = getMsec(logitude);
		parameters.put("longitude", Long.toString(msec));

		String latitude = tokens[5];
		msec = getMsec(latitude);
		parameters.put("latitude", Long.toString(msec));

		String fixType = tokens[6];
		parameters.put("fixType", fixType);

		String speed = tokens[7];
		parameters.put("speed", speed);

		String course = tokens[8];
		parameters.put("course", course);
		
		String satellites = tokens[9];
		parameters.put("sats", satellites);

		String hdop = tokens[10];
		if(!hdop.isEmpty())
		{
			parameters.put("hdop", hdop);
		}

		String vdop = tokens[11];
		if(!vdop.isEmpty())
		{
			parameters.put("vdop", vdop);
		}

		String height = tokens[12];
		parameters.put("altitude", height);

		String rmc = GPSPosition.createNMEArecord("RMC", parameters);
		String gga = GPSPosition.createNMEArecord("GGA", parameters);

		eventParameters.put("rmc", rmc);
		eventParameters.put("gga", gga);
	}    
    
	static final long MSEC_FACTOR = 60*60*1000 / 10000;

	long getMsec(String position)
	{
		return Long.parseLong(position) * MSEC_FACTOR;
	}
    
}

