package ercogener

import de.proveo.event.util.efm.groovy.EventParser;
import java.util.*;
import java.text.*;
import org.apache.commons.lang.StringUtils;
import de.proveo.util.geo.GPSPosition;

/**
 *
 * @author <a href="mailto:jbader@zebra.com">Joachim Bader</a>
 * $Rev: 22427 $
 */
class ErcoGenerEventParser implements EventParser 
{
    /**
     * 
     * @param event the event send form the device
     * @param parsedEventParameters map, should be filled with parsed key value information
     * @param parsedMetaInfo map, should be filled with parsed event meta information
     */
    public void parseEvent(Object event, Map parsedEventParameters, Map metaInfo)
    {
        String eventStr = (String) event;
        if(eventStr.startsWith('\$GPLOC'))
        {
            parseGPLOC(eventStr, parsedEventParameters, metaInfo);
			
			metaInfo.put("hwsn", parsedEventParameters.get("imei"));
			metaInfo.put("hwType", "erco&gener");			
        }
    }
    
    /**
     * Configuraiton command:
	 * AT+LOCFRT=2,"ID","NS","HD","AL","CS","SK","GS","IP","DI","OW","AN","DT"
	 * 
     * Example record:
	 * $GPLOC,354112023002277,A,1,081110.00,4907.53327,N,01001.88600,E,8,0.96,452.6,0.00,0.668,47.8,01000,78.7,000000000000,NNNN,3173002929954408,240811*02
     */
    void parseGPLOC(String eventStr, Map parsedEventParameters, Map metaInfo)
    {
        String[] tokens = StringUtils.splitByWholeSeparatorPreserveAllTokens(eventStr, ",");
                
        if(!tokens[0].equals('\$GPLOC'))
        {
            throw new IllegalArgumentException("not a GPLOC record");
        }
        
		// ID
		parsedEventParameters.put("imei", tokens[1]);
		
		// ST (A: correct Values   V: Warning)
		parsedEventParameters.put("navigationReceiverWarning", tokens[2]);
		
		// FX (-1: Unable   0: Invalid  1: GPS Fix (2D/3D)   2: DGPS Fix
		parsedEventParameters.put("fixIndicator", Integer.parseInt(tokens[3]));

		// TM
		String time = tokens[4];

		// LT
		String angleStr = tokens[5];
		
		// LI
		String indicator = tokens[6];		
		parsedEventParameters.put("latitude", getPositionAngle(angleStr, indicator));

		// LG
		angleStr = tokens[7];
		
		// LU
		indicator = tokens[8];
		parsedEventParameters.put("longitude", getPositionAngle(angleStr, indicator));

		// NS
		if(!tokens[9].isEmpty())
		{
			parsedEventParameters.put("sats", Integer.parseInt(tokens[9]));
		}
		
		// HD
		if(!tokens[10].isEmpty())
		{		
			parsedEventParameters.put("hdop", Double.parseDouble(tokens[10]));
		}

		// AL
		if(!tokens[11].isEmpty())
		{
			parsedEventParameters.put("altitude", parseDouble(tokens[11]));
		}		
		
		// CS
		if(!tokens[12].isEmpty())
		{
			parsedEventParameters.put("course", parseDouble(tokens[12]));
		}
		
		// SK
		if(!tokens[13].isEmpty())
		{		
			parsedEventParameters.put("speed", Double.parseDouble(tokens[13]));
		}
		
		// GS: Geoid
		if(!tokens[14].isEmpty())
		{		
			parsedEventParameters.put("geoid", Double.parseDouble(tokens[14]));
		}
		
		
		// IP
		parseInputs(parsedEventParameters, tokens[15]);
		
		// DI
		parsedEventParameters.put("distance", Double.parseDouble(tokens[16]));
		
		// OW
		parsedEventParameters.put("iButton", tokens[17]);
		
		
		// AN
		parseAnalogInputs(parsedEventParameters, tokens[18], tokens[19]);
				
		// DT
		String[] lastTwoFields = StringUtils.splitByWholeSeparatorPreserveAllTokens(tokens[20], "*");		
		String date = lastTwoFields[0];
		String checksum = lastTwoFields[1];
		
		parsedEventParameters.put("timestamp", getTimestamp(date+time+" UTC"));
		parsedEventParameters.put("time", parsedEventParameters.get("timestamp"));

		filterEmptyValues(parsedEventParameters);

		String rmc = GPSPosition.createNMEArecord("RMC", convertToString(parsedEventParameters));
		String gga = GPSPosition.createNMEArecord("GGA", convertToString(parsedEventParameters));

		parsedEventParameters.put("rmc", rmc);
		parsedEventParameters.put("gga", gga);        
    }
    
	Map convertToString(Map map)
	{
		Map<String, String> strMap = new HashMap<String, String>();
		for(Map.Entry<String, Object> entry: map.entrySet())
		{
			strMap.put(entry.getKey(), entry.getValue().toString());
		}

		return strMap;
	}
    
    void parseInputs(Map parsedEventParameters, String inputs) {
		for(int index=0;index<inputs.length();index++)
        {
            boolean value = inputs.getAt(inputs.length()-index-1)=='1'?true:false;
            parsedEventParameters.put("in"+index, value);
        }
	}

	void parseAnalogInputs(Map parsedEventParameters, String statusFlags, String value) {
		for(int index=0;index<4;index++)
		{
			String key;
			switch(index)
			{
				case 0:
					key = "gps.antenna.power"; break;
				case 1:
					key = "an0"; break;
				case 2:
					key = "an1"; break;
				case 3:
					key = "powersupply"; break;						
			}
			
			String flag = statusFlags.substring(index, index+1);
			
			parsedEventParameters.put(key+".status", flag);			
			parsedEventParameters.put(key, Integer.parseInt(value.substring(index*4,(index+1)*4)));
		}
	}    
    	
	int getPositionAngle(String angleStr, String indicator)
	{
		int separatorIndex = angleStr.indexOf('.');
		separatorIndex -= 2;
		
		if(separatorIndex<0)
		{
			return 0;
		}

		String degrees = angleStr.substring(0, separatorIndex);
		String minutes = angleStr.substring(separatorIndex);

		long angle = (long) (Double.parseDouble(minutes)*60.0*1000.0);
		angle += Integer.parseInt(degrees)*60*60*1000;

		if(indicator.equals("S") || indicator.equals("W"))
		{
			angle *= -1;
		}

		return angle;
	}
    
    void filterEmptyValues(Map parsedEventParameters) {
		Iterator<Map.Entry<String, Object>> it = parsedEventParameters.entrySet().iterator();
		while(it.hasNext())
		{
			Map.Entry entry = it.next();
			if((entry.getValue()==null) || (entry.getValue().toString().isEmpty()))
			{
				it.remove();
			}
		}
	}    
        
    static final DateFormat dateFormat = new SimpleDateFormat("ddMMyyHHmmss.SSS zzz");
    
	long getTimestamp(String timestamp) throws Exception
	{
        return dateFormat.parse(timestamp + " UTC").getTime();
	}    
	
	double parseDouble(String str)
	{
		return Double.parseDouble(str);
	}
}

